// Copyright (C) 2017-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_LUA_PROXY_HH
#define NV_LUA_PROXY_HH

#include <nv/common.hh>
#include <nv/core/types.hh>
#include <nv/stl/string.hh>
#include <nv/stl/string_table.hh>
#include <nv/lua/lua_values.hh>

namespace nv
{
	namespace lua
	{
		class state;

		class stack_proxy
		{
		public:
			stack_proxy( state* state, sint32 index ) : m_state( state ), m_index( index ) {}
			stack_proxy( const stack_proxy& ) = delete;
			stack_proxy& operator=( const stack_proxy& ) = delete;
			stack_proxy( stack_proxy&& ) = default;
			stack_proxy& operator=( stack_proxy&& ) = default;

			uint32 get_uint32( uint32 def = 0 ) const;
			sint32 get_sint32( sint32 def = 0 ) const;
			char get_char( char def = ' ' ) const;
			f64 get_f64( f64 def = 0.0 ) const;
			f32 get_f32( f32 def = 0.0f ) const;
			bool get_bool( bool def = false ) const;

			template< typename E >
			E get_enum( E def = E() ) const
			{
				return static_cast<E>( get_uint32( static_cast<sint32>( def ) ) );
			}

			template< typename T >
			T as( const T& def ) const
			{
				return static_cast<T>( pass_traits< converted_type_t<T> >::to( *m_state, m_index, def ) );
			}

			template< typename T >
			T as() const
			{
				return static_cast<T>( pass_traits< converted_type_t<T> >::to( *m_state, m_index ) );
			}

			bool is_table() const;
			bool is_number() const;
			bool is_bool() const;
			bool is_string() const;
			bool is_valid() const;
			bool is_nil() const { return !is_valid(); }
			operator bool() const { return is_valid(); }

			const_string get_string() const { return const_string( get_string_view() ); }
			const_string as_string()        { return const_string( as_string_view() ); }
			string32 get_string32() const   { return string32( get_string_view() ); }
			string32 as_string32()          { return string32( as_string_view() ); }
			string64 get_string64() const   { return string64( get_string_view() ); }
			string64 as_string64()          { return string64( as_string_view() ); }
			string128 get_string128() const { return string128( get_string_view() ); }
			string128 as_string128()        { return string128( as_string_view() ); }
			shash64 get_shash64() const     { return is_string() ? shash64( get_string_view() ) : shash64(); }
			shash64 as_shash64()            { return is_string() ? shash64( as_string_view() ) : shash64(); }

			shash64 get_string( string_table* table ) const
			{
				string_view result( get_string_view() );
				return table ? table->insert( result ) : result;
			}

			shash64 as_string( string_table* table )
			{
				string_view result( as_string_view() );
				return table ? table->insert( result ) : result;
			}

			template < typename T >
			bool read( T& t ) const
			{
				const type_database* tdb = get_type_db();
				NV_ASSERT( tdb, "stack_proxy::read - type database not set!" );
				const type_entry* entry = tdb->get_type<T>();
				return read( entry, &t );
			}
			bool read( const type_entry* entry, void* object ) const;
// bool write?

			// TO FUCKING DO : non-copyable stringview
//			string_view get_string_view() const;
//			string_view to_string_view();
		protected:
			const type_database* get_type_db() const;

			// Not public because not permanent
			string_view get_string_view() const;
			// Not public because not permanent
			string_view as_string_view();

			state* m_state;
			sint32 m_index;
			friend class table_guard;
		};

		class temporary_proxy : public stack_proxy
		{
		protected:
			temporary_proxy( const temporary_proxy& ) = delete;
			temporary_proxy& operator=( const temporary_proxy& ) = delete;
			temporary_proxy( temporary_proxy&& ) = default;
			temporary_proxy& operator=( temporary_proxy&& ) = default;

			temporary_proxy( state* state );
		public:
			~temporary_proxy();
			friend class table_guard;
		};



		class kv_proxy
		{
		public:
			kv_proxy( state* state ) : m_state( state ) {}
			kv_proxy( const kv_proxy& ) = delete;
			kv_proxy& operator=( const kv_proxy& ) = delete;
			kv_proxy( kv_proxy&& ) = default;
			kv_proxy& operator=( kv_proxy&& ) = default;
			const stack_proxy key() const { return stack_proxy( m_state, -2 ); }
			stack_proxy value() const { return stack_proxy( m_state, -1 ); }
		protected:
			state* m_state;
		};

	}

}

#endif // NV_LUA_PROXY_HH
