// Copyright (C) 2017-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file component_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_COMPONENT_HH
#define NV_ECS_COMPONENT_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/priority_queue.hh>
#include <nv/stl/range.hh>
#include <nv/core/types.hh>
#include <nv/ecs/component_storage.hh>

namespace nv
{
	namespace lua
	{
		class stack_proxy;
	}

	namespace ecs
	{

		template < 
			typename Ecs,
			typename Component,
			typename Handler,
			template < typename, typename > class IndexTable = flat_index_table
		>
		class component : public Ecs::component_interface
		{
		public:
			typedef Ecs                                     ecs_type;
			typedef typename ecs_type::message              message_type;
			typedef typename ecs_type::handle_type          handle_type;
			typedef typename ecs_type::time_type            time_type;
			typedef IndexTable< handle_type, sint32 >       index_table_type;
			typedef index_table< handle_type, sint32 >      idx_table;
			typedef Component                               value_type;
			typedef Component                               component_type;
			typedef component_storage_handler< value_type > storage_type;
//			typedef vector< value_type > storage_type;
			typedef typename index_table_type::index_type   index_type;

			typedef typename storage_type::iterator         iterator;
			typedef typename storage_type::const_iterator   const_iterator;
			typedef typename storage_type::reference        reference;
			typedef typename storage_type::const_reference  const_reference;

			component( ecs_type& a_ecs, string_view a_name, bool relational = false ) 
				: m_ecs( a_ecs ), m_relational( relational ), m_storage( nullptr )
			{
				m_index = new index_table_type;
				m_storage = (storage_type*)m_ecs.template register_component<component_type, Handler>( a_name, this );
// 				if ( reserve != 0 )
// 				{
// 					m_storage->reserve( reserve );
// 				}
			}

			void temp_storage( storage_type* storage )
			{

			}

			virtual uint32 temp_index( handle_type h ) const  final
			{
				return m_index->get( h );
			}


			inline value_type& insert( handle_type h )
			{
				index_type i = m_index->insert( h );
				NV_ASSERT( i == index_type( m_storage->size() ), "Fail!" );
				NV_UNUSED( i );
				m_storage->emplace_back();
				return m_storage->back();
			}

			template < typename ...Args >
			value_type& insert( handle_type h, Args&&... args )
			{
				index_type i = m_index->insert( h );
				NV_ASSERT( i == index_type( m_storage->size() ), "Fail!" );
				NV_UNUSED( i );
				m_storage->emplace_back( nv::forward<Args>( args )... );
				return m_storage->back();
			}

			virtual void* insert_raw( handle_type h ) final
			{
				return &insert( h );
			}


			bool exists( handle_type h )
			{
				return m_index->exists( h );
			}

			void call_destroy( value_type* data )
			{
				for ( auto dh : this->m_destroy )
					dh( data );
			}

			virtual void clear_() final
			{
				m_index->clear();
				for ( uint32 i = 0; i < m_storage->size(); ++i )
					call_destroy( &( *m_storage )[i] );
				m_storage->clear();
			}

			value_type* get( handle_type h )
			{
				index_type i = m_index->get( h );
				return i >= 0 ? &( ( *m_storage )[unsigned( i )] ) : nullptr;
			}
			const value_type* get( handle_type h ) const
			{
				index_type i = m_index->get( h );
				return i >= 0 ? &( ( *m_storage )[unsigned( i )] ) : nullptr;
			}

			void* get_raw( handle_type h ) { return get( h ); }
			const void* get_raw( handle_type h ) const { return get( h ); }

			virtual void remove_( handle_type h ) final
			{
				value_type* v = get( h );
				if ( v == nullptr ) return;
				call_destroy( v );
				index_type dead_eindex = m_index->remove_swap( h );
				if ( dead_eindex == -1 ) return;
				if ( dead_eindex != static_cast<index_type>( m_storage->size() - 1 ) )
				{
					( *m_storage )[unsigned( dead_eindex )] = move( m_storage->back() );
				}
				m_storage->pop_back();
				if ( m_relational )
					relational_rebuild( dead_eindex );
			}

			void remove_by_index_( uint32 i )
			{
				if ( i > m_storage->size() ) return;
				call_destroy( &(*m_storage)[i] );
				index_type dead_eindex = m_index->remove_swap_by_index( i );
				if ( dead_eindex == -1 ) return;
				if ( dead_eindex != static_cast<index_type>( m_storage->size() - 1 ) )
				{
					( *m_storage )[unsigned( dead_eindex )] = move( m_storage->back() );
				}
				m_storage->pop_back();
				if ( m_relational )
					relational_rebuild( dead_eindex );
			}

			virtual void on_attach( handle_type, handle_type child ) final
			{
				if ( m_relational )
				{
					relational_recursive_rebuild( child );
				}
			}

			virtual ~component()
			{
				//clear_();
				delete m_index;
			}
			
			inline size_t size() const { return m_storage->size(); }
 			inline virtual component_storage* storage() { return m_storage; }
		protected:
			ecs_type&        m_ecs;
			storage_type*    m_storage;
		private:
			void swap( handle_type a, handle_type b )
			{
				index_type ai = m_index->get( a );
				index_type bi = m_index->get( b );
				if ( ai < 0 || bi < 0 ) return;
				m_index->swap( a, b );
				value_type t = nv::move( ( *m_storage )[ai] );
				( *m_storage )[ai] = nv::move( ( *m_storage )[bi] );
				( *m_storage )[bi] = nv::move( t );
			}

			handle_type extract_owner( index_type i )
			{
				return *(handle_type*)( &( ( *m_storage )[i] ) );
			}

			void relational_rebuild( index_type i )
			{
				handle_type h = extract_owner( i );
				handle_type p = m_ecs.get_parent( h );
				if ( !p ) return;
				if ( i < m_index->get( p ) )
				{
					swap( h, p );
					relational_rebuild( i );
				}
			}

			void relational_recursive_rebuild( handle_type h )
			{
				handle_type p = m_ecs.get_parent( h );
				if ( !p ) return;
				if ( m_index->get( h ) < m_index->get( p ) )
				{
					swap( h, p );
					for ( auto c : m_ecs.children( h ) )
						relational_recursive_rebuild( c );
				}
			}

			bool       m_relational;
			idx_table* m_index;
		};

	}

}

#endif // NV_ECS_COMPONENT_HH
