// Copyright (C) 2015-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file capi.hh
 * @author Kornel Kisielewicz
 * @brief C API functions
 *
 * This header is temporary
 */

// TODO: sanity checking with NV_DEBUG_ASSERT

#ifndef NV_BASE_CAPI_HH
#define NV_BASE_CAPI_HH

#include <nv/base/common.hh>

namespace nv
{

	namespace detail
	{

#if NV_COMPILER == NV_MSVC
#define NV_CAPI_CALL(name) detail::name
		extern "C" {
			NV_NOALIAS NV_RESTRICT void * __cdecl calloc( size_t _Count, size_t _Size );
			NV_NOALIAS             void   __cdecl free( void * _Memory );
			NV_NOALIAS NV_RESTRICT void * __cdecl malloc( size_t );
			NV_NOALIAS NV_RESTRICT void * __cdecl realloc( void * _Memory, size_t _NewSize );
			NV_NOALIAS NV_RESTRICT void * __cdecl _aligned_malloc( size_t _Size, size_t _Alignment );
			NV_NOALIAS             void   __cdecl _aligned_free( void* _Block );
//			NV_NOALIAS NV_RESTRICT void * __cdecl _aligned_realloc( void*  _Block, size_t _Size, size_t _Alignment );
			void *  __cdecl memcpy( void * _Dst, const void * _Src, size_t _Size );
			void *  __cdecl memmove( void * _Dst, const void * _Src, size_t _Size );
			void *  __cdecl memset( void * _Dst, int _Val, size_t _Size );
			void *  __cdecl memchr( const void * _Buf, int _Val, size_t _MaxCount );
			int     __cdecl memcmp( const void * _Buf1, const void * _Buf2, size_t _Size );
			int     __cdecl strcmp( const char * _Str1, const char * _Str2 );
			int     __cdecl strncmp( const char * _Str1, const char * _Str2, size_t _MaxCount );
			size_t  __cdecl strlen( const char * _Str );
		}
#endif 
		// TODO: remove when clang 3.5 will be used (with builtins)
#if NV_COMPILER == NV_CLANG || NV_COMPILER == NV_GNUC
#define NV_CAPI_CALL(name) detail::name
		extern "C" {
			extern void * __cdecl calloc( nv::size_t, nv::size_t );
			extern void   __cdecl free( void * );
			extern void * __cdecl malloc( nv::size_t );
			extern void * __cdecl realloc( void * , nv::size_t );
			extern void * __cdecl aligned_alloc( nv::size_t, nv::size_t );
			extern void * __cdecl memcpy( void * , const void * , nv::size_t );
			extern void * __cdecl memmove( void * , const void * , nv::size_t );
			extern void * __cdecl memset( void * , int , nv::size_t );
			extern void * __cdecl memchr( const void * , int , nv::size_t );
			extern int    __cdecl memcmp( const void * , const void * , nv::size_t  );
			extern int    __cdecl strcmp ( const char * , const char * );
			extern int    __cdecl strncmp( const char * , const char * , size_t );
			extern nv::size_t __cdecl strlen ( const char * );
		}
#endif
	}

	inline void* nvmalloc( size_t size )
	{
		return NV_CAPI_CALL(malloc)( size );
	}
	inline void* nvrealloc( void* p, size_t size )
	{
		if ( size > 0 ) return NV_CAPI_CALL( realloc )( p, size );
		NV_CAPI_CALL( free )( p );
		return nullptr;
	}
	inline void nvfree( void* p )
	{
		if ( p ) NV_CAPI_CALL( free )( p );
	}

	inline void* nvaligned_malloc( size_t size, size_t alignment )
	{
#if NV_COMPILER == NV_MSVC
		return NV_CAPI_CALL( _aligned_malloc )( size, alignment );
#else
		return NV_CAPI_CALL( aligned_alloc )( size, alignment );
#endif
	}
	inline void nvaligned_free( void* p )
	{
#if NV_COMPILER == NV_MSVC
		if ( p ) NV_CAPI_CALL( _aligned_free )( p );
#else
		if ( p ) NV_CAPI_CALL( free )( p );
#endif
	}


	inline void* nvmemcpy( void *dest, const void *src, size_t count )
	{
		if ( dest == src || count == 0 ) return dest;
		NV_ASSERT( dest && src, "Bad parameter to nvmemcpy!" );
		return NV_CAPI_CALL( memcpy )( dest, src, count );
	}

	inline void* nvmemmove( void *dest, const void *src, size_t count )
	{
		if ( dest == src || count == 0 ) return dest;
		NV_ASSERT( dest && src, "Bad parameter to nvmemmove!" );
		return NV_CAPI_CALL( memmove )( dest, src, count );
	}

	inline void* nvmemset( void *dest, unsigned char value, size_t count )
	{
		NV_ASSERT( dest, "Bad parameter to nvmemset!" );
		return NV_CAPI_CALL( memset )( dest, static_cast<int>( value ), count );
	}

	inline void* nvmemchr( const void *src, unsigned char value, size_t max_count )
	{
		NV_ASSERT( src, "Bad parameter to nvmemchr!" );
		return NV_CAPI_CALL( memchr )( src, static_cast<int>( value ), max_count );
	}

	inline int nvmemcmp( const void * m1, const void * m2, nv::size_t max_count )
	{
		NV_ASSERT( m1 && m2, "Bad parameter to nvmemcmp!" );
		return NV_CAPI_CALL( memcmp )( m1, m2, max_count );
	}

	inline int nvstrcmp( const char * s1, const char * s2 )
	{
		NV_ASSERT( s1 && s2, "Bad parameter to nvstrcmp!" );
		return NV_CAPI_CALL( strcmp )( s1, s2 );
	}	

	inline int nvstrncmp( const char * s1, const char * s2, size_t max_count )
	{
		return NV_CAPI_CALL( strncmp )( s1, s2, max_count );
	}

	inline nv::size_t nvstrlen( const char * s )
	{
		return NV_CAPI_CALL( strlen )( s );
	}

}

#undef NV_CAPI_CALL

#endif // NV_BASE_CAPI_HH
