// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file geometric.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief geometric math ops
*/

#ifndef NV_STL_MATH_GEOMERTIC_HH
#define NV_STL_MATH_GEOMERTIC_HH

#include <nv/stl/math/common.hh>
#include <nv/stl/math/basic.hh>
#include <nv/stl/math/exponential.hh>

namespace nv
{

	namespace math
	{

		namespace detail
		{
			template < typename T >
			struct dot_impl {};

			template < typename T >
			struct dot_impl< tvec2< T > >
			{
				inline static T get( const tvec2<T>& a, const tvec2<T>& b )
				{
					tvec2<T> tmp( a * b );
					return tmp.x + tmp.y;
				}
			};

			template < typename T >
			struct dot_impl< tvec3< T > >
			{
				inline static T get( const tvec3<T>& a, const tvec3<T>& b )
				{
					tvec3<T> tmp( a * b );
					return tmp.x + tmp.y + tmp.z;
				}
			};

			template < typename T >
			struct dot_impl< tvec4< T > >
			{
				inline static T get( const tvec4<T>& a, const tvec4<T>& b )
				{
					tvec4<T> tmp( a * b );
					return ( tmp.x + tmp.y ) + ( tmp.z + tmp.w );
				}
			};
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T length( T x )
		{
			return abs( x );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T inv_length( T v )
		{
			return T( 1 ) / abs( x );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline value_type_t<T> length( const T& v )
		{
			return nv::sqrt( dot( v, v ) );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline value_type_t<T> inv_length( const T& v )
		{
			return T( 1 ) / nv::sqrt( dot( v, v ) );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline value_type_t<T> length_sq( const T& v )
		{
			return dot( v, v );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T distance( T a, T b )
		{
			return length( b - a );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline value_type_t<T> distance( const T& a, const T& b )
		{
			return length( b - a );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T dot( T a, T b )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return a * b;
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline value_type_t<T> dot( const T& a, const T& b )
		{
			return detail::dot_impl< T >::get( a, b );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline tvec3<T> cross( const tvec3<T>& a, const tvec3<T>& b )
		{
			return tvec3<T>(
				a.y * b.z - b.y * a.z,
				a.z * b.x - b.z * a.x,
				a.x * b.y - b.x * a.y
			);
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T normalize( T x )
		{
			return x < T( 0 ) ? T( -1 ) : T( 1 );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline T normalize( const T& x )
		{
			return x * inversesqrt( dot( x, x ) );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T faceforward( T n, T i, T nref )
		{
			return dot( nref, i ) < static_cast<T>( 0 ) ? n : -n;
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline T faceforward( const T& n, const T& i, const T& nref )
		{
			return dot( nref, i ) < static_cast<value_type_t<T>>( 0 ) ? n : -n;
		}

		template < typename T, typename enable_if< is_fp<T>::value >::type* = nullptr >
		inline T reflect( const T& i, const T& n )
		{
			return i - n * dot( n, i ) * static_cast<T>( 2 );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T refract( const T& i, const T& n, const T& eta )
		{
			const T dotv( dot( n, i ) );
			const T k( static_cast<T>( 1 ) - eta * eta * ( static_cast<T>( 1 ) - dotv * dotv ) );
			return ( eta * i - ( eta * dotv + sqrt( k ) ) * n ) * static_cast<T>( k >= static_cast<T>( 0 ) );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline T refract( const T& i, const T& n, value_type_t<T> eta )
		{
			typedef value_type_t<T> value_type;

			value_type const dotv( dot( n, i ) );
			value_type const k( static_cast<T>( 1 ) - eta * eta * ( static_cast<value_type>( 1 ) - dotv * dotv ) );
			return ( eta * i - ( eta * dotv + sqrt( k ) ) * n ) * static_cast<value_type>( k >= static_cast<value_type>( 0 ) );
		}

	}
}

#endif // NV_STL_MATH_GEOMERTIC_HH
