// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file copy.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief algorithm copy
*/

#ifndef NV_STL_ALGORITHM_COPY_HH
#define NV_STL_ALGORITHM_COPY_HH

#include <nv/stl/algorithm/common.hh>
#include <nv/stl/algorithm/raw.hh>
#include <nv/stl/iterator.hh>
#include <nv/stl/type_traits/properties.hh>

namespace nv
{

	namespace detail
	{
		template< typename InputIterator, typename OutputIterator, typename AnyType >
		inline OutputIterator copy_impl( InputIterator first, InputIterator last, OutputIterator out, AnyType, input_iterator_tag, forward_iterator_tag )
		{
			for ( ; first != last; ++first, ++out )
				*first = *out;
			return out;
		}

		template< typename BlockAccessIterator, typename OutputIterator >
		inline BlockAccessIterator copy_impl( BlockAccessIterator first, BlockAccessIterator last, OutputIterator out, true_type, block_access_iterator_tag, block_access_iterator_tag )
		{
			return raw_alias_copy( first, last, out );
		}

		template< typename InputIterator, typename OutputIterator, typename AnyType >
		inline OutputIterator copy_n_impl( InputIterator first, size_t n, OutputIterator out, AnyType, input_iterator_tag, forward_iterator_tag )
		{
			for ( ; n-- > 0; ++first, ++out ) *out = *first++;
			return out;
		}

		template< typename BlockAccessIterator, typename OutputIterator >
		inline BlockAccessIterator copy_n_impl( BlockAccessIterator first, BlockAccessIterator n, OutputIterator out, true_type, block_access_iterator_tag, block_access_iterator_tag )
		{
			return raw_alias_copy_n( first, n, out );
		}
	}

	template< class InputIterator, class OutputIterator >
	inline OutputIterator copy( InputIterator first, InputIterator last, OutputIterator out )
	{
		typedef typename iterator_traits< InputIterator >::value_type value_type;
		typedef typename iterator_traits< InputIterator >::iterator_category input_iterator_category;
		typedef typename iterator_traits< OutputIterator >::iterator_category output_iterator_category;
		return detail::copy_impl( first, last, out,
			has_trivial_copy<value_type>::value, input_iterator_category(), output_iterator_category() );
	}

	template< class InputIterator, class OutputIterator >
	inline OutputIterator copy_n( InputIterator first, size_t n, OutputIterator out )
	{
		typedef typename iterator_traits< InputIterator >::value_type value_type;
		typedef typename iterator_traits< InputIterator >::iterator_category input_iterator_category;
		typedef typename iterator_traits< OutputIterator >::iterator_category output_iterator_category;
		return detail::copy_n_impl( first, n, out,
			has_trivial_copy<value_type>::value, input_iterator_category(), output_iterator_category() );
	}

}

#endif // NV_STL_ALGORITHM_COPY_HH
