// Copyright (C) 2017-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/lua/lua_proxy.hh"

#include "nv/lua/lua_raw.hh"
#include "nv/lua/lua_state.hh"
#include "nv/lua/lua_types.hh"

using namespace nv;
using namespace nv::lua;

uint32 nv::lua::stack_proxy::get_uint32( uint32 def ) const
{
	return lua_type( *m_state, m_index ) == LUA_TNUMBER ? nlua_tounsigned( *m_state, m_index ) : def;
}

sint32 nv::lua::stack_proxy::get_sint32( sint32 def ) const
{
	return lua_type( *m_state, m_index ) == LUA_TNUMBER ? static_cast<sint32>( lua_tointeger( *m_state, m_index ) ) : def;
}

char nv::lua::stack_proxy::get_char( char def /*= ' ' */ ) const
{
	return ( lua_type( *m_state, m_index ) == LUA_TSTRING && nlua_rawlen( *m_state, m_index ) > 0 ) ? lua_tostring( *m_state, m_index )[0] : def;
}

nv::f64 nv::lua::stack_proxy::get_f64( f64 def /*= 0.0 */ ) const
{
	return lua_type( *m_state, m_index ) == LUA_TNUMBER ? lua_tonumber( *m_state, m_index ) : def;
}

nv::f32 nv::lua::stack_proxy::get_f32( f32 def /*= 0.0f */ ) const
{
	return lua_type( *m_state, m_index ) == LUA_TNUMBER ? static_cast<float>( lua_tonumber( *m_state, m_index ) ) : def;
}

bool nv::lua::stack_proxy::get_bool( bool def /*= false */ ) const
{
	return lua_type( *m_state, m_index ) == LUA_TBOOLEAN ? lua_toboolean( *m_state, m_index ) != 0 : def;
}

bool nv::lua::stack_proxy::is_number() const
{
	return lua_type( *m_state, m_index ) == LUA_TNUMBER;
}

bool nv::lua::stack_proxy::is_bool() const
{
	return lua_type( *m_state, m_index ) == LUA_TBOOLEAN;
}

bool nv::lua::stack_proxy::is_string() const
{
	return lua_type( *m_state, m_index ) == LUA_TSTRING;
}

bool nv::lua::stack_proxy::is_valid() const
{
	return !( lua_isnil( *m_state, m_index ) );
}

bool nv::lua::stack_proxy::read( const type_entry* entry, void* object ) const
{
	NV_ASSERT_ALWAYS( m_state->get_type_data()->get_type_database() == entry->type_db, "Type database mismatch between Lua and entry!" );
	if ( lua_type( *m_state, m_index ) != LUA_TTABLE ) return false;
	return nv::lua::read_rtti_type( m_state, entry, object, m_index );
}

bool nv::lua::stack_proxy::is_table() const
{
	return lua_type( *m_state, m_index ) == LUA_TTABLE;
}

nv::string_view nv::lua::stack_proxy::get_string_view() const
{
	size_t l = 0;
	const char* str = nullptr;
	if ( lua_type( *m_state, m_index ) == LUA_TSTRING )
	{
		str = lua_tolstring( *m_state, m_index, &l );
	}
	return string_view( str, static_cast<uint32>( l ) );
}

nv::string_view nv::lua::stack_proxy::as_string_view()
{
	size_t l = 0;
	const char* str = nullptr;
	str = lua_tolstring( *m_state, m_index, &l );
	return string_view( str, static_cast<uint32>( l ) );
}

nv::lua::temporary_proxy::temporary_proxy( state* state )
	: stack_proxy( state, lua_gettop( *state ) )
{

}

nv::lua::temporary_proxy::~temporary_proxy()
{
	lua_pop( *m_state, 1 );
}
