// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file vec3.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief vec3
 */

#ifndef NV_STL_MATH_VEC3_HH
#define NV_STL_MATH_VEC3_HH

#include <nv/stl/math/common.hh>

namespace nv
{

	namespace math
	{
		template <typename T>
		struct tvec3
		{
			typedef tvec3<T> type;
			typedef T value_type;
			static constexpr uint32 SIZE = 3;

			union
			{
				struct { T x, y, z; };
				struct { T r, g, b; };
				struct { T s, t, p; };
			};

			typedef uint32 size_type;
			inline constexpr size_type size() const { return 3; }

			inline T & operator[]( size_type i )
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return ( &x )[i];
			}

			inline T operator[]( size_type i ) const
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return ( &x )[i];
			}


			inline tvec3()
				: x( static_cast<T>( 0 ) )
				, y( static_cast<T>( 0 ) )
				, z( static_cast<T>( 0 ) )
			{}

			inline tvec3( const tvec3<T> & m ) = default;
			inline tvec3( tvec3<T> && m ) = default;
			inline tvec3<T> & operator=( const tvec3<T> & m ) = default;
			inline tvec3<T> & operator=( tvec3<T> && m ) = default;

			inline explicit tvec3( T scalar )
				: x( scalar )
				, y( scalar )
				, z( scalar )
			{}

			inline explicit tvec3( no_init_t ) {}

			inline tvec3( T sx, T sy, T sz )
				: x( sx )
				, y( sy )
				, z( sz )
			{}

			template <typename A, typename B, typename C>
			inline tvec3( A sx, B sy, C sz )
				: x( static_cast<T>( sx ) )
				, y( static_cast<T>( sy ) )
				, z( static_cast<T>( sz ) )
			{}

			template <typename A, typename B>
			inline explicit tvec3( const tvec2<A> &a, B b )
				: x( static_cast<T>( a.x ) )
				, y( static_cast<T>( a.y ) )
				, z( static_cast<T>( b ) )
			{}

			template <typename A, typename B>
			inline tvec3( A a, const tvec2<B> & b )
				: x( static_cast<T>( a ) )
				, y( static_cast<T>( b.x ) )
				, z( static_cast<T>( b.y ) )
			{}

			template <typename U>
			inline explicit tvec3( const tvec4<U> & v )
				: x( static_cast<T>( v.x ) )
				, y( static_cast<T>( v.y ) )
				, z( static_cast<T>( v.z ) )
			{}

			template <typename U>
			inline explicit tvec3( const tvec3<U> & v )
				: x( static_cast<T>( v.x ) )
				, y( static_cast<T>( v.y ) )
				, z( static_cast<T>( v.z ) )
			{}

			inline tvec3<T> & operator+=( T scalar )
			{
				this->x += static_cast<T>( scalar );
				this->y += static_cast<T>( scalar );
				this->z += static_cast<T>( scalar );
				return *this;
			}

			inline tvec3<T> & operator+=( const tvec3<T> & v )
			{
				this->x += static_cast<T>( v.x );
				this->y += static_cast<T>( v.y );
				this->z += static_cast<T>( v.z );
				return *this;
			}

			inline tvec3<T> & operator-=( T scalar )
			{
				this->x -= static_cast<T>( scalar );
				this->y -= static_cast<T>( scalar );
				this->z -= static_cast<T>( scalar );
				return *this;
			}

			inline tvec3<T> & operator-=( const tvec3<T> & v )
			{
				this->x -= static_cast<T>( v.x );
				this->y -= static_cast<T>( v.y );
				this->z -= static_cast<T>( v.z );
				return *this;
			}

			inline tvec3<T> & operator*=( T scalar )
			{
				this->x *= static_cast<T>( scalar );
				this->y *= static_cast<T>( scalar );
				this->z *= static_cast<T>( scalar );
				return *this;
			}

			inline tvec3<T> & operator*=( const tvec3<T> & v )
			{
				this->x *= static_cast<T>( v.x );
				this->y *= static_cast<T>( v.y );
				this->z *= static_cast<T>( v.z );
				return *this;
			}

			inline tvec3<T> & operator/=( T v )
			{
				this->x /= static_cast<T>( v );
				this->y /= static_cast<T>( v );
				this->z /= static_cast<T>( v );
				return *this;
			}

			inline tvec3<T> & operator/=( const tvec3<T> & v )
			{
				this->x /= static_cast<T>( v.x );
				this->y /= static_cast<T>( v.y );
				this->z /= static_cast<T>( v.z );
				return *this;
			}

			inline tvec3<T> & operator++()
			{
				++this->x;
				++this->y;
				++this->z;
				return *this;
			}

			inline tvec3<T> & operator--()
			{
				--this->x;
				--this->y;
				--this->z;
				return *this;
			}

			inline tvec3<T> operator++( int )
			{
				tvec3<T> result( *this );
				++*this;
				return result;
			}

			inline tvec3<T> operator--( int )
			{
				tvec3<T> result( *this );
				--*this;
				return result;
			}

		};

		template <typename T>
		inline tvec3<T> operator-( const tvec3<T> & v )
		{
			return tvec3<T>(
				-v.x,
				-v.y,
				-v.z );
		}

		template <typename T>
		inline tvec3<T> operator+( const tvec3<T> & v, T scalar )
		{
			return tvec3<T>(
				v.x + scalar,
				v.y + scalar,
				v.z + scalar );
		}

		template <typename T>
		inline tvec3<T> operator+( T scalar, const tvec3<T> & v )
		{
			return tvec3<T>(
				scalar + v.x,
				scalar + v.y,
				scalar + v.z );
		}

		template <typename T>
		inline tvec3<T> operator+( const tvec3<T> & v1, const tvec3<T> & v2 )
		{
			return tvec3<T>(
				v1.x + v2.x,
				v1.y + v2.y,
				v1.z + v2.z );
		}

		template <typename T>
		inline tvec3<T> operator-( const tvec3<T> & v, T scalar )
		{
			return tvec3<T>(
				v.x - scalar,
				v.y - scalar,
				v.z - scalar );
		}

		template <typename T>
		inline tvec3<T> operator-( T scalar, const tvec3<T> & v )
		{
			return tvec3<T>(
				scalar - v.x,
				scalar - v.y,
				scalar - v.z );
		}

		template <typename T>
		inline tvec3<T> operator-( const tvec3<T> & v1, const tvec3<T> & v2 )
		{
			return tvec3<T>(
				v1.x - v2.x,
				v1.y - v2.y,
				v1.z - v2.z );
		}

		template <typename T>
		inline tvec3<T> operator*( const tvec3<T> & v, T scalar )
		{
			return tvec3<T>(
				v.x * scalar,
				v.y * scalar,
				v.z * scalar );
		}

		template <typename T>
		inline tvec3<T> operator*( T scalar, const tvec3<T> & v )
		{
			return tvec3<T>(
				scalar * v.x,
				scalar * v.y,
				scalar * v.z );
		}

		template <typename T>
		inline tvec3<T> operator*( const tvec3<T> & v1, const tvec3<T> & v2 )
		{
			return tvec3<T>(
				v1.x * v2.x,
				v1.y * v2.y,
				v1.z * v2.z );
		}

		template <typename T>
		inline tvec3<T> operator/( const tvec3<T> & v, T scalar )
		{
			return tvec3<T>(
				v.x / scalar,
				v.y / scalar,
				v.z / scalar );
		}

		template <typename T>
		inline tvec3<T> operator/( T scalar, const tvec3<T> & v )
		{
			return tvec3<T>(
				scalar / v.x,
				scalar / v.y,
				scalar / v.z );
		}

		template <typename T>
		inline tvec3<T> operator/( const tvec3<T> & v1, const tvec3<T> & v2 )
		{
			return tvec3<T>(
				v1.x / v2.x,
				v1.y / v2.y,
				v1.z / v2.z );
		}

		template <typename T>
		inline bool operator==( const tvec3<T> & v1, const tvec3<T> & v2 )
		{
			return ( v1.x == v2.x ) && ( v1.y == v2.y ) && ( v1.z == v2.z );
		}

		template <typename T>
		inline bool operator!=( const tvec3<T> & v1, const tvec3<T> & v2 )
		{
			return ( v1.x != v2.x ) || ( v1.y != v2.y ) || ( v1.z != v2.z );
		}

	}

}

#endif // NV_STL_MATH_VEC3_HH
