// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file model_manager.hh
* @author Kornel Kisielewicz
* @brief model_manager
*/

#ifndef NV_ENGINE_MODEL_MANAGER_HH
#define NV_ENGINE_MODEL_MANAGER_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/core/random.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/engine/material_manager.hh>
#include <nv/engine/mesh_manager.hh>
#include <nv/engine/animation.hh>
#include <nv/engine/resource_system.hh>

#define MNODE 1

namespace nv
{
	enum class model_node_choice
	{
		ALL,
		WEIGHTED,
		PATTERN_CHECKER
	};

	enum model_node_flags
	{
		MNF_FORCE,
		MNF_STATIC,
	};

	enum class phx_shape
	{
		BOX,
		CYLINDER,
	};


	struct model;

	struct ragdoll_data;

	struct model_node
	{
		resource< data_channel_set > mesh;
		resource< material >         material;
		string32                     tag;
		transform                    local;
		random_range< vec3 >         position;
		random_range< vec3 >         rotation;
		vector< model_node* >        children;
		sint16                       attach_id = -1;
		uint16                       weight    = 0;
		float                        chance    = 1.0f;
		flags<16,uint16>             nflags;
		vec3                         phx_hextents;
		vec3                         phx_offset;
		float                        phx_mass  = 0.0f;
		phx_shape                    phx_shape = phx_shape::BOX;
		model_node_choice            choice    = model_node_choice::ALL;

		void clone_values_to( model_node* target ) const
		{
			target->mesh      = mesh;
			target->material  = material;
			target->tag       = tag;
			target->local     = local;
			target->rotation  = rotation;
			target->position  = position;

			target->phx_hextents = phx_hextents;
			target->phx_offset   = phx_offset;
			target->phx_mass     = phx_mass;
			target->phx_shape    = phx_shape;
			target->attach_id = attach_id;
			target->weight    = weight;
			target->chance    = chance;
			target->nflags    = nflags;
			target->choice    = choice;
		}

		~model_node()
		{
			for ( auto c : children ) delete c;
		}
	};

	struct model : model_node
	{
		resource< animator_data >      animator;
		resource< animator_bind_data > bind_data;
		resource< data_channel_set >   phx_mesh;
		// TODO: change to resource
		string32                       ragdoll_id;
		transform                      root;
		shash64                        attach;
		flags< 32 >                    flags;
 	};

	struct flat_model_element
	{
		resource< data_channel_set > mesh;
		resource< material >         material;
		shash64                      tag;
		sint16                       attach_id;
		sint16                       parent_id;
		transform					 local;
		vec3                         phx_hextents;
		vec3                         phx_offset;
		float                        phx_mass;
		phx_shape                    phx_shape;
		uint32                       gflags;
	};

	struct flat_model
	{
		flat_model_element             elements[32];
		resource< data_channel_set >   phx_mesh;
		uint32                         count;
		shash64                        attach;
		transform					   local;
		flags< 32 >                    flags;
	};

	NV_RTTI_DECLARE_NAME( model_node_choice, "model_node_choice" )
	NV_RTTI_DECLARE_NAME( model_node_flags, "model_node_flags" )
	NV_RTTI_DECLARE_NAME( model, "model" )
	NV_RTTI_DECLARE_NAME( phx_shape, "phx_shape" )

	enum flat_model_flags
	{
		FMF_SELECTED    = 0x01,
		FMF_FOCUS       = 0x02,
		FMF_FAIL_CHANCE = 0x04,
		FMF_FAIL_CHOICE = 0x08,
		FMF_RANGE_GHOST = 0x10,
	};

	enum flatten_flags
	{
		FF_GENERATE_CHANCE  = 0x02,
		FF_GENERATE_CHOICE  = 0x04,
		FF_GENERATE_GHOST   = 0x08,
		FF_GENERATE_NLOCAL  = 0x10,
		FF_GENERATE_STATIC  = 0x20,
		FF_GENERATE_NSTATIC = 0x40,
	};

	class model_manager : public lua_resource_manager< model >
	{
	public:
		model_manager( resource_manager* rm, animator_bind_manager* binds, mesh_data_manager* mesh_datas )
			: m_rm( rm )
			, m_animator_binds( binds )
			, m_mesh_datas( mesh_datas )
		{
		}
		virtual string_view get_storage_name() const { return "models"; }
		virtual string_view get_resource_name() const { return "model"; }
		static flat_model flatten( 
			const model* m, 
			random_base& rng, 
			ivec2 control,
			vector< const model_node* >* map = nullptr, 
			uint32 gen_flags = 0, 
			const model_node* select = nullptr )
		{
			flat_model result;
			if ( !m ) return result;
			result.attach = m->attach;
			result.local = m->root;
			result.flags = m->flags;
			result.count = 0;
			result.phx_mesh = m->phx_mesh;
			flatten( result, m, rng, control, transform(), transform(), -1, map, gen_flags, 0, select );
			return result;
		}
	protected:
		static void flatten( 
			flat_model& result, 
			const model_node* m, 
			random_base& rng, 
			ivec2 control,
			const transform& ptr, 
			const transform& fptr,
			sint16 parent_id,
			vector< const model_node* >* map,
			uint32 gen_flags,
			uint32 parent_flags,
			const model_node* selected
		)
		{
			if ( m->chance < 1.0f )
				if ( rng.frand() > m->chance )
				{
					if ( gen_flags & FF_GENERATE_CHANCE )
						parent_flags |= FMF_FAIL_CHANCE;
					else
						return;
				}

			bool skip_static = ( gen_flags & FF_GENERATE_NSTATIC ) && m->nflags[MNF_STATIC];
			bool parse_mesh = ( ( m->mesh || !m->tag.empty() ) && !skip_static ) || ( m->nflags[MNF_FORCE] );

			transform tr = fptr * m->local;
			if ( !(gen_flags & FF_GENERATE_NLOCAL) )
				tr = ptr * tr;


			vec3 position = rng.eval( m->position );
			vec3 rotation = rng.eval( m->rotation );
			transform rtr = transform( position, nv::quat( rotation ) );
			tr = tr * rtr;

			if ( m == selected ) parent_flags |= FMF_FOCUS;

			if ( gen_flags & FF_GENERATE_STATIC )
				if ( !m->nflags[MNF_STATIC] )
					return;

			transform ftr;
			if ( !parse_mesh && ( gen_flags & FF_GENERATE_NLOCAL ) )
			{
				ftr = tr;
			}


			if ( parse_mesh )
			{
				uint32 id = result.count++;
				NV_ASSERT_ALWAYS( id < 32, "flat_model node limit reached!" );
				flat_model_element& re = result.elements[id];
				re.mesh = m->mesh;
				re.material = m->material;
				re.local = tr;
				re.tag = m->tag;
				re.parent_id = parent_id;
				re.attach_id = m->attach_id;
				re.gflags = parent_flags;
				re.phx_hextents = m->phx_hextents;
				re.phx_offset   = m->phx_offset;
				re.phx_mass     = m->phx_mass;
				re.phx_shape    = m->phx_shape;

				if ( map ) map->push_back( m );
				parent_id = sint16(id);

				if ( m == selected ) 
				{
					re.gflags |= FMF_SELECTED;
					if ( gen_flags & FF_GENERATE_GHOST && 
							( m->position.max != m->position.min || 
							m->rotation.max != m->rotation.min ) )
					{
						uint32 min_id = result.count++;
						flat_model_element& min_re = result.elements[min_id];
						min_re.mesh = m->mesh;
						min_re.material = m->material;
						min_re.local = ( ptr * m->local ) * transform( m->position.min, nv::quat( m->rotation.min ) );
						min_re.parent_id = parent_id;
						min_re.attach_id = m->attach_id;
						min_re.gflags = parent_flags | FMF_RANGE_GHOST;

						uint32 max_id = result.count++;
						flat_model_element& max_re = result.elements[max_id];
						max_re.mesh = m->mesh;
						max_re.material = m->material;
						max_re.local = ( ptr * m->local ) * transform( m->position.max, nv::quat( m->rotation.max ) );
						max_re.parent_id = parent_id;
						max_re.attach_id = m->attach_id;
						max_re.gflags = parent_flags | FMF_RANGE_GHOST;
					}
				}
			}

			if ( m->choice != model_node_choice::ALL )
			{
				model_node* pick = nullptr;
				if ( m->choice == model_node_choice::WEIGHTED )
				{
					uint16 total_weight = 0;
					for ( auto c : m->children )
						total_weight += c->weight;
					if ( total_weight == 0 ) return;
					sint32 roll = rng.srand( total_weight );
					for ( auto c : m->children )
					{
						roll -= c->weight;
						if ( roll < 0 ) { pick = c; break; }
					}
				}
				else if ( m->choice == model_node_choice::PATTERN_CHECKER )
				{
					pick = m->children[( control.x + control.y ) % m->children.size()];
				}

				if ( pick == nullptr ) return;
				if ( gen_flags & FF_GENERATE_CHOICE )
					for ( auto c : m->children )
					{
						uint32 flags = parent_flags;
						if ( c != pick ) flags |= FMF_FAIL_CHOICE;
						flatten( result, c, rng, control, tr, ftr, parent_id, map, gen_flags, flags, selected );
					}
				else
					flatten( result, pick, rng, control, tr, ftr, parent_id, map, gen_flags, parent_flags, selected );
			}
			else
				for ( auto c : m->children )
				{
					flatten( result, c, rng, control, tr, ftr, parent_id, map, gen_flags, parent_flags, selected );
				}
		}

	protected:
		virtual bool load_resource( lua::table_guard& table, shash64 id );
		void read_model_node( lua::table_guard& table, model_node* node, resource< mesh_data > def_data );
	private:
		resource_manager*      m_rm;
		animator_bind_manager* m_animator_binds;
		mesh_data_manager*     m_mesh_datas;
	};

}

#endif // NV_ENGINE_MODEL_MANAGER_HH
