// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file resource_manager.hh
 * @author Kornel Kisielewicz
 * @brief engine resource manager
 */

#ifndef NV_ENGINE_RESOURCE_MANAGER_HH
#define NV_ENGINE_RESOURCE_MANAGER_HH

#include <nv/common.hh>
#include <nv/interface/context.hh>
#include <nv/stl/file_system.hh>
#include <nv/core/resource.hh>
#include <nv/lua/lua_state.hh>
#include <nv/stl/hash_store.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/string.hh>

namespace nv
{

	template < typename T, bool Heap = true >
	struct resource_storage_policy;

	template < typename T >
	struct resource_storage_policy< T, true >
	{
		typedef T* type;
		static void free( T* value ) { delete value; }
		static T* to_pointer( T* value ) { return value; }
		static T* to_stored( void* value ) { return reinterpret_cast<T*>( value ); }
	};

	template < typename T >
	struct resource_storage_policy< T, false >
	{
		typedef T type;
		static void free( T ) {}
		static T* to_pointer( T& value ) { return &value; }
		static T to_stored( void* value ) { return *reinterpret_cast<T*>( value ); }
	};

	class resource_system;

	class lua_resource_manager_base : public resource_handler
	{
	public:
		lua_resource_manager_base( resource_type_id id ) : resource_handler( id ), m_lua( nullptr ) {}
		virtual void initialize( lua::state* state );
		virtual string_view get_storage_name() const = 0;
		virtual string_view get_resource_name() const = 0;
		virtual void clear() = 0;
		void load_all( bool do_clear = true );
		void preload_ids();
		virtual bool load_resource( const string_view& id );
		virtual ~lua_resource_manager_base() {}
	protected:
		virtual bool load_resource( lua::table_guard& table, shash64 id ) = 0;
		hash_store< shash64, string64 > m_id_hash;
		lua::state* m_lua;
	};

	template < typename T, bool Heap = true, typename Base = resource_handler >
	class custom_resource_manager : public Base
	{
	public:
		typedef resource_storage_policy< T, Heap > policy_type;
		typedef Base                       base_type;
		typedef T                          value_type;
		typedef resource< T >              resource_type;
		typedef typename policy_type::type stored_type;

		custom_resource_manager() : base_type( resource_type_id( rtti_type_hash<T>::hash() ) ) {}
		resource_type get( const string_view& id )
		{
			if ( exists( id ) )
				return this->template create< T >( id );
			else
			{
				if ( this->load_resource( id ) )
				{
					return this->template create< T >( id );
				}
			}
			NV_LOG_ERROR( "resource_manager.get(\"",id,"\") failed!" );
			return resource_type();
		}

// 		resource_type get_( uint64 id )
// 		{
// 			if ( exists( shash64( id ) ) ) return this->template create< T >( shash64( id ) );
// 			NV_LOG_ERROR( "resource_manager.get(\"", id, "\") failed!" );
// 			return resource_type();
// 		}

		virtual void clear()
		{
			for ( auto data : m_store )
			{
				release( data.second );
				policy_type::free( data.second );
			}
			m_store.clear();
		}

		virtual ~custom_resource_manager()
		{
			clear();
		}
	protected:

		virtual void remove( resource_id id )
		{
			auto m = m_store.find( shash64( id ) );
			if ( m != m_store.end() )
			{
				release( m->second );
				m_store.erase( shash64( id ) );
			}
		}

		virtual void rename( resource_id id, resource_id new_id )
		{
			auto m = m_store.find( shash64( id ) );
			if ( m != m_store.end() )
			{
				auto mdata = m->second;
				m_store.erase( shash64( id ) );
				add( new_id, mdata );
			}
		}

		resource_type add( shash64 id, stored_type resource )
		{
			auto m = m_store.find( shash64( id ) );
			if ( m != m_store.end() )
			{
				release( m->second );
			}
			m_store[id] = resource;
			return this->template create< T >( id );
		}

		virtual bool exists( resource_id id )
		{
			return m_store.find( shash64( id ) ) != m_store.end();
		}

		virtual void raw_add( resource_id id, void* value )
		{
			add( id, policy_type::to_stored( value ) );
		}


		virtual const void* raw_lock( resource_id id, bool )
		{
			auto m = m_store.find( id );
			return m != m_store.end() ? policy_type::to_pointer( m->second ) : nullptr;
		}

		virtual void unlock( resource_id ) {}
		virtual void release( resource_id ) {}
		virtual void release( stored_type ) {}

		hash_store< shash64, stored_type > m_store;
	};

	template < typename T, bool Heap = true >
	using lua_resource_manager = custom_resource_manager< T, Heap, lua_resource_manager_base >;


	template < typename T, bool Heap = true, typename Base = resource_handler >
	class manual_resource_manager : public custom_resource_manager< T, Heap, Base >
	{
	public:
		manual_resource_manager() {}
		using custom_resource_manager< T, Heap, Base >::add;
	};

	template < typename T, typename SourceManager, bool Heap = true >
	class dependant_resource_manager : public manual_resource_manager< T, Heap >
	{
	public:
		typedef SourceManager source_manager_type;
		typedef typename SourceManager::resource_type source_type;
	public:
		explicit dependant_resource_manager( source_manager_type* source ) : m_source( source ) {}
		virtual resource< T > load_resource( source_type u )
		{
			if ( exists( u.id() ) ) return this->template create< T >( u.id() );
			return create_resource( u );
		}
	protected:
		virtual resource< T > create_resource( source_type u ) = 0;
		virtual bool load_resource( const string_view& id )
		{
			source_type u = m_source->get( id );
			return u && create_resource( u );
		}
	protected:
		source_manager_type* m_source;
	};

	template < typename T, bool Heap = true, typename Base = resource_handler >
	class file_resource_manager : public manual_resource_manager< T, Heap, Base >
	{
	public:
		explicit file_resource_manager() {}
		void add_base_path( const string_view& path )
		{
			m_paths.emplace_back( path );
		}
	protected:
		stream* open_stream( file_system& fs, const string_view& filename )
		{
			if ( fs.exists( filename ) ) return fs.open( filename );
			if ( m_paths.size() > 0 )
			{
				for ( const auto& path : m_paths )
				{
					string128 fpath( path );
					fpath.append( filename );
					if ( fs.exists( fpath ) )
						return fs.open( fpath );
				}
			}
			return nullptr;
		}

		vector< string128 > m_paths;
	};

}

#endif // NV_ENGINE_RESOURCE_MANAGER_HH
