// Copyright (C) 2017-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file component_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_COMPONENT_STORAGE_HH
#define NV_ECS_COMPONENT_STORAGE_HH

#include <nv/common.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/handle_manager.hh>
#include <nv/stl/hash_map.hh>

namespace nv
{

	namespace ecs
	{

		class component_storage
		{
		protected:
			component_storage() {}
			template < typename T >
			void initialize()
			{
				m_data = nullptr;
				m_size = 0;
				m_allocated = 0;
				// TODO: detect trivially destructible objects
				m_destructor = raw_destroy_object < T >;
				m_csize = sizeof( T );
			}
		public:
			void reserve( uint32 count )
			{
				reallocate( count );
			}
			uint32 size() const { return m_size; }
			uint32 raw_size() const { return m_size * m_csize; }
			void reset()
			{
				clear();
				nvfree( m_data );
				m_data = nullptr;
				m_allocated = 0;
			}
			void clear()
			{
				uint32 count = m_size;
				uint8* d     = m_data;
				for ( ; count > 0; --count, d += m_csize )
					m_destructor(d);
				m_size = 0;
			}
			void* raw() { return m_data; }
			const void* raw() const { return m_data; }
			// TODO: Debug-time type checking
			template < typename T >
			T* as() { return (T*)m_data; }
			template < typename T >
			const T* as() const { return m_data; }

			void pop_back()
			{
				NV_ASSERT( m_size > 0, "BAD OP!" );
				m_size--;
				m_destructor( m_data + m_size * m_csize );
			}

			~component_storage()
			{
				reset();
			}
		protected:
			void grow()
			{
				uint32 new_size = m_size + 1;
				if ( new_size > m_allocated )
				{
					reallocate( m_allocated > 3 ? m_allocated * 2 : 8 );
				}
				m_size = new_size;
			}

			void reallocate( uint32 new_size )
			{
				m_data = static_cast<uint8*>( nvrealloc( m_data, new_size * m_csize ) );
				m_allocated = new_size;
				NV_ASSERT( m_data, "OMG" );
			}

			//thash64 m_type;
			uint32       m_csize;
			uint32       m_allocated;
			uint32       m_size;
			destructor_t m_destructor;
			uint8*       m_data;
		};

		template < typename Component >
		class component_storage_handler : public component_storage
		{
		public:
			typedef Component         value_type;
			typedef Component*        iterator;
			typedef const Component*  const_iterator;
			typedef Component&        reference;
			typedef const Component&  const_reference;

			component_storage_handler() 
			{
				initialize<Component>();
			}
			Component* data() { return (Component*)m_data; }
			const Component* data() const { return (Component*)m_data; }
			inline const Component& operator[] ( uint32 i ) const { return ((Component*)m_data)[i]; }
			inline Component& operator[] ( uint32 i ) { return ( (Component*)m_data )[i]; }

			inline iterator        begin() { return (Component*)m_data; }
			inline const_iterator  begin()  const { return (const Component*)m_data; }
			inline void*           raw_begin() { return m_data; }
			inline const void*     raw_begin() const { return m_data; }

			inline iterator        end() { return ((Component*)m_data)+m_size; }
			inline const_iterator  end()  const { return ( (Component*)m_data ) + m_size; }
			inline void*           raw_end() { return ( (Component*)m_data ) + m_size; }
			inline const void*     raw_end() const { return ( (Component*)m_data ) + m_size; }

			template < typename... Args >
			void emplace_back( Args&&... args )
			{
				grow();
				construct_object( data() + sint32( m_size ) - 1, forward<Args>( args )... );
			}

			Component& back()
			{
				NV_ASSERT( m_size > 0, "EMPTY COMPONENT STORAGE" );
				return *( data() + sint32( m_size ) - 1 );
			}
		};

		template < typename Component >
		component_storage_handler< Component >* storage_cast( component_storage* storage )
		{
			// TODO: error checking
			return ( component_storage_handler< Component >* )storage;
		}

		template < typename Handle, typename Component >
		class index_storage 
		{
		public:
			typedef Handle                                 handle_type;
			typedef Component                              value_type;
			typedef vector< value_type >                   storage_type;
			typedef sint32                                 index_type;
			typedef typename storage_type::iterator        iterator;
			typedef typename storage_type::const_iterator  const_iterator;
			typedef typename storage_type::reference       reference;
			typedef typename storage_type::const_reference const_reference;

			explicit index_storage( uint32 reserve = 0 )
			{
				if ( reserve != 0 )
				{
					m_data.reserve( reserve );
				}
			}

			inline value_type& insert( handle_type h )
			{
				insert_index( h );
				m_data.emplace_back();
				return m_data.back();
			}

			template < typename ...Args >
			inline value_type& insert( handle_type h, Args&&... args )
			{
				insert_index( h );
				m_data.emplace_back( nv::forward<Args>( args )... );
				return m_data.back();
			}

			inline bool exists( handle_type h )
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return false;
				return m_indexes[h.index()] >= 0;
			}

			value_type* get( handle_type h )
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return nullptr;
				index_type i = m_indexes[h.index()];
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			const value_type* get( handle_type h ) const
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return nullptr;
				index_type i = m_indexes[h.index()];
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			void clear()
			{
				m_handles.clear();
				m_indexes.clear();
				m_data.clear();
			}

			void remove( handle_type h )
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() || m_indexes[h.index()] == -1 )
					return;
				handle_type swap_handle = m_handles.back();
				index_type  dead_eindex = m_indexes[h.index()];
				if ( dead_eindex != static_cast<index_type>( m_handles.size() - 1 ) )
				{
					m_handles[unsigned( dead_eindex )] = swap_handle;
					m_indexes[swap_handle.index()] = dead_eindex;
					m_data[unsigned( dead_eindex )] = move( m_data.back() );
				}
				m_handles.pop_back();
				m_data.pop_back();
				m_indexes[h.index()] = -1;

			}

			inline handle_type get_handle( index_type i ) const { return m_handles[unsigned( i )]; }

			inline const value_type& operator[] ( index_type i ) const { return m_data[i]; }
			inline value_type& operator[] ( index_type i ) { return m_data[i]; }

			inline size_t size() const { return m_data.size(); }

			inline iterator        begin() { return m_data.begin(); }
			inline const_iterator  begin()  const { return m_data.cbegin(); }

			inline iterator        end() { return m_data.end(); }
			inline const_iterator  end()  const { return m_data.cend(); }

		protected:
			void insert_index( handle_type h )
			{
				NV_ASSERT( !exists( h ), "Reinserting handle!" );
				resize_indexes_to( index_type( h.index() ) );
				index_type lindex = m_handles.size();
				m_indexes[h.index()] = index_type( lindex );
				m_handles.push_back( h );
				NV_ASSERT( lindex == index_type( m_data.size() ), "Fail!" );
			}

			void resize_indexes_to( index_type i )
			{
				index_type size = index_type( m_indexes.size() );
				if ( i >= size )
				{
					if ( size == 0 ) size = 1;
					while ( i >= size ) size = size * 2;
					m_indexes.resize( static_cast<uint32>( size ), -1 );
				}
			}

			vector< handle_type > m_handles;
			vector< index_type >  m_indexes;
			vector< value_type >  m_data;
		};

		template < typename Handle, typename Component >
		class index_multi_storage
		{
		public:
			typedef Handle                                 handle_type;
			typedef Component                              value_type;
			typedef vector< value_type >                   storage_type;
			typedef index_table< handle_type >             index_table_type;
			typedef typename index_table_type::index_type  index_type;
			typedef typename storage_type::iterator        iterator;
			typedef typename storage_type::const_iterator  const_iterator;
			typedef typename storage_type::reference       reference;
			typedef typename storage_type::const_reference const_reference;

			explicit index_multi_storage( uint32 reserve = 0 )
			{
				if ( reserve != 0 )
				{
					m_data.reserve( reserve );
				}
			}

			inline value_type& insert( handle_type h )
			{
				insert_index( h );
				m_data.emplace_back();
				return m_data.back();
			}

			template < typename ...Args >
			inline value_type& insert( handle_type h, Args&&... args )
			{
				insert_index( h );
				m_data.emplace_back( nv::forward<Args>( args )... );
				return m_data.back();
			}

			inline bool exists( handle_type h )
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return false;
				return m_indexes[h.index()] >= 0;
			}

			value_type* get( handle_type h )
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return nullptr;
				index_type i = m_indexes[h.index()];
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			const value_type* get( handle_type h ) const
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return nullptr;
				index_type i = m_indexes[h.index()];
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			void clear()
			{
				m_entries.clear();
				m_indexes.clear();
				m_data.clear();
			}

// 			void remove( handle_type h )
// 			{
// 				if ( h.is_nil() || h.index() >= m_indexes.size() || m_indexes[h.index()] == -1 )
// 					return;
// 
// 				// this can be done more efficiently
// 				while ( m_indexes[h.index()] >= 0 )
// 				{
// 					remove_component( h.index() );
// 				}
// 			}

			inline handle_type get_handle( index_type i ) const { return m_entries[unsigned( i )].handle; }

			inline const value_type& operator[] ( index_type i ) const { return m_data[i]; }
			inline value_type& operator[] ( index_type i ) { return m_data[i]; }

			inline size_t size() const { return m_data.size(); }

			inline iterator        begin() { return m_data.begin(); }
			inline const_iterator  begin()  const { return m_data.cbegin(); }

			inline iterator        end() { return m_data.end(); }
			inline const_iterator  end()  const { return m_data.cend(); }

		protected:

//  			void swap_to_end( handle_type h )
//  			{
// 				NV_ASSERT( !h.is_nil() && h.index() < m_indexes.size() && m_indexes[h.index()] != -1, "Bad parameter passed" );
// 				handle_type swap_handle = m_handles.back();
// 				index_type  dead_eindex = m_indexes[h.index()];
// 				if ( dead_eindex != static_cast<index_type>( m_handles.size() - 1 ) )
// 				{
// 					m_handles[unsigned( dead_eindex )] = swap_handle;
// 					m_indexes[swap_handle.index()]     = dead_eindex;
// 					m_data[unsigned( dead_eindex )]    = move( m_data.back() );
// 				}
// 				m_indexes[h.index()] = -1;
//  			}

			void insert_index( handle_type h )
			{
				NV_ASSERT( h, "Bad handle!" );
				index_type i = index_type( h.index() )
				resize_indexes_to( i );
				index_type lindex = m_entries.size();
				m_entries.push_back( h );
				index_type next = m_indexes[h.index()];
				if ( next >= 0 ) // exists
				{
					entry_type& e = m_entries[next];
					e.prev = lindex;
				}
				m_indexes[h.index()] = index_type( lindex );
				m_entries.back().next = next;
				NV_ASSERT( lindex == index_type( m_data.size() ), "Fail!" );
			}

			void resize_indexes_to( index_type i )
			{
				index_type size = index_type( m_indexes.size() );
				if ( i >= size )
				{
					if ( size == 0 ) size = 1;
					while ( i >= size ) size = size * 2;
					m_indexes.resize( static_cast<uint32>( size ), -1 );
				}
			}

			struct entry_type
			{
				handle_type handle;
				index_type  next;
				index_type  prev;

				entry_type() : handle(), next( -1 ), prev( -1 ) {}
				entry_type( handle_type h ) : handle( h ), next( -1 ), prev( -1 ) {}
			};

			vector< entry_type > m_entries;
			vector< index_type > m_indexes;
			vector< value_type > m_data;
		};

		class direct_storage;
		/*
		template < typename Handle, typename Component >
		class hash_storage
		{
		public:
			typedef Handle                                 handle_type;
			typedef Component                              value_type;
			typedef vector< value_type >                   storage_type;
			typedef typename storage_type::iterator        iterator;
			typedef typename storage_type::const_iterator  const_iterator;
			typedef typename storage_type::reference       reference;
			typedef typename storage_type::const_reference const_reference;
		
			explicit hash_storage( uint32 reserve = 0 )
			{
				if ( reserve != 0 )
				{
					m_data.reserve( reserve );
				}
			}

			inline value_type& insert( handle_type h )
			{
				auto result = m_data.insert_key( h );
				NV_ASSERT( result.second, "Reinsert of handle!" );
				m_data.emplace_back();
				return m_data.back();
			}

			template < typename ...Args >
			inline value_type& insert( handle_type h, Args&&... args )
			{
				insert_index( h );
				m_data.emplace_back( nv::forward<Args>( args )... );
				return m_data.back();
			}

			inline bool exists( handle_type h )
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return false;
				return m_indexes[h.index()] >= 0;
			}

			value_type* get( handle_type h )
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return nullptr;
				index_type i = m_indexes[h.index()];
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			const value_type* get( handle_type h ) const
			{
				if ( h.is_nil() || h.index() >= m_indexes.size() ) return nullptr;
				index_type i = m_indexes[h.index()];
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			void clear()
			{
				m_data.clear();
			}

			void remove( handle_type h )
			{


			}

			inline handle_type get_handle( sint32 i ) const { return m_handles[unsigned( i )]; }

			inline const value_type& operator[] ( index_type i ) const { return m_data[i]; }
			inline value_type& operator[] ( index_type i ) { return m_data[i]; }

			inline size_t size() const { return m_data.size(); }

			inline iterator        begin() { return m_data.begin(); }
			inline const_iterator  begin()  const { return m_data.cbegin(); }

			inline iterator        end() { return m_data.end(); }
			inline const_iterator  end()  const { return m_data.cend(); }

		protected:
			vector< value_type > m_data;
		};


		class hash_multi_storage;
		*/
	}

}

#endif // NV_ECS_COMPONENT_STORAGE_HH
