// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file vec2.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief vec2
 */

#ifndef NV_STL_MATH_VEC2_HH
#define NV_STL_MATH_VEC2_HH

#include <nv/stl/math/common.hh>

namespace nv
{

	namespace math
	{

		template <typename T>
		struct tvec2
		{
			typedef tvec2<T> type;
			typedef T value_type;

			static constexpr uint32 SIZE = 2;

			union
			{
				struct { T x, y; };
				struct { T r, g; };
				struct { T s, t; };
			};

			typedef uint32 size_type;
			inline constexpr size_type size() const { return 2; }

			inline T & operator[]( size_type i )
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return ( &x )[i];
			}

			inline T operator[]( size_type i ) const
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return ( &x )[i];
			}

			inline tvec2()
				: x( static_cast<T>( 0 ) )
				, y( static_cast<T>( 0 ) )
			{} 

			inline tvec2( const tvec2<T> & m ) = default;
			inline tvec2( tvec2<T> && m ) = default;
			inline tvec2<T> & operator=( const tvec2<T> & m ) = default;
			inline tvec2<T> & operator=( tvec2<T> && m ) = default;

			inline explicit tvec2( T scalar )
				: x( scalar )
				, y( scalar )
			{}

			inline explicit tvec2( no_init_t ) {}

			inline tvec2( T sx, T sy )
				: x( sx )
				, y( sy )
			{}

			template < typename A, typename B >
			inline tvec2( A sx, B sy )
				: x( static_cast<T>( sx ) )
				, y( static_cast<T>( sy ) )
			{}

			template <typename U>
			inline explicit tvec2( const tvec3<U> & v )
				: x( static_cast<T>( v.x ) )
				, y( static_cast<T>( v.y ) )
			{}

			template <typename U>
			inline explicit tvec2( const tvec4<U> & v )
				: x( static_cast<T>( v.x ) )
				, y( static_cast<T>( v.y ) )
			{}

			template <typename U>
			inline explicit tvec2( const tvec2<U> & v )
				: x( static_cast<T>( v.x ) )
				, y( static_cast<T>( v.y ) )
			{}

			inline tvec2<T> & operator+=( T scalar )
			{
				this->x += static_cast<T>( scalar );
				this->y += static_cast<T>( scalar );
				return *this;
			}

			inline tvec2<T> & operator+=( const tvec2<T> & v )
			{
				this->x += static_cast<T>( v.x );
				this->y += static_cast<T>( v.y );
				return *this;
			}

			inline tvec2<T> & operator-=( T scalar )
			{
				this->x -= static_cast<T>( scalar );
				this->y -= static_cast<T>( scalar );
				return *this;
			}

			inline tvec2<T> & operator-=( const tvec2<T> & v )
			{
				this->x -= static_cast<T>( v.x );
				this->y -= static_cast<T>( v.y );
				return *this;
			}

			inline tvec2<T> & operator*=( T scalar )
			{
				this->x *= static_cast<T>( scalar );
				this->y *= static_cast<T>( scalar );
				return *this;
			}

			inline tvec2<T> & operator*=( const tvec2<T> & v )
			{
				this->x *= static_cast<T>( v.x );
				this->y *= static_cast<T>( v.y );
				return *this;
			}

			inline tvec2<T> & operator/=( T scalar )
			{
				this->x /= static_cast<T>( scalar );
				this->y /= static_cast<T>( scalar );
				return *this;
			}

			inline tvec2<T> & operator/=( const tvec2<T> & v )
			{
				this->x /= static_cast<T>( v.x );
				this->y /= static_cast<T>( v.y );
				return *this;
			}

			inline tvec2<T> & operator++()
			{
				++this->x;
				++this->y;
				return *this;
			}

			inline tvec2<T> & operator--()
			{
				--this->x;
				--this->y;
				return *this;
			}

			inline tvec2<T> operator++( int )
			{
				tvec2<T> result( *this );
				++*this;
				return result;
			}

			inline tvec2<T> operator--( int )
			{
				tvec2<T> result( *this );
				--*this;
				return result;
			}

		};


		template <typename T>
		inline tvec2<T> operator-( const tvec2<T> & v )
		{
			return tvec2<T>(
				-v.x,
				-v.y );
		}

		template <typename T>
		inline tvec2<T> operator+( const tvec2<T> & v, T scalar )
		{
			return tvec2<T>(
				v.x + scalar,
				v.y + scalar );
		}

		template <typename T>
		inline tvec2<T> operator+( T scalar, const tvec2<T> & v )
		{
			return tvec2<T>(
				scalar + v.x,
				scalar + v.y );
		}

		template <typename T>
		inline tvec2<T> operator+( const tvec2<T> & v1, const tvec2<T> & v2 )
		{
			return tvec2<T>(
				v1.x + v2.x,
				v1.y + v2.y );
		}

		template <typename T>
		inline tvec2<T> operator-( const tvec2<T> & v, T scalar )
		{
			return tvec2<T>(
				v.x - scalar,
				v.y - scalar );
		}

		template <typename T>
		inline tvec2<T> operator-( T scalar, const tvec2<T> & v )
		{
			return tvec2<T>(
				scalar - v.x,
				scalar - v.y );
		}

		template <typename T>
		inline tvec2<T> operator-( const tvec2<T> & v1, const tvec2<T> & v2 )
		{
			return tvec2<T>(
				v1.x - v2.x,
				v1.y - v2.y );
		}

		template <typename T>
		inline tvec2<T> operator*( const tvec2<T> & v1, T v2 )
		{
			return tvec2<T>(
				v1.x * v2,
				v1.y * v2 );
		}

		template <typename T>
		inline tvec2<T> operator*( T scalar, const tvec2<T> & v )
		{
			return tvec2<T>(
				scalar * v.x,
				scalar * v.y );
		}

		template <typename T>
		inline tvec2<T> operator*( const tvec2<T> & v1, const tvec2<T> & v2 )
		{
			return tvec2<T>(
				v1.x * v2.x,
				v1.y * v2.y );
		}

		template <typename T>
		inline tvec2<T> operator/( const tvec2<T> & v, T scalar )
		{
			return tvec2<T>(
				v.x / scalar,
				v.y / scalar );
		}

		template <typename T>
		inline tvec2<T> operator/( T scalar, const tvec2<T> & v )
		{
			return tvec2<T>(
				scalar / v.x,
				scalar / v.y );
		}

		template <typename T>
		inline tvec2<T> operator/( const tvec2<T> & v1, const tvec2<T> & v2 )
		{
			return tvec2<T>(
				v1.x / v2.x,
				v1.y / v2.y );
		}

		template <typename T>
		inline bool operator==( const tvec2<T> & v1, const tvec2<T> & v2 )
		{
			return ( v1.x == v2.x ) && ( v1.y == v2.y );
		}

		template <typename T>
		inline bool operator!=( const tvec2<T> & v1, const tvec2<T> & v2 )
		{
			return ( v1.x != v2.x ) || ( v1.y != v2.y );
		}

	}

}

#endif // NV_STL_MATH_VEC2_HH
