// Copyright (C) 2012-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file image_data.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Image data struct
 */

#ifndef NV_INTERFACE_IMAGE_DATA_HH
#define NV_INTERFACE_IMAGE_DATA_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/algorithm/raw.hh>

namespace nv
{

	enum pixel_format : uint8
	{
		RGB8,
		RGBA8,
		BGR8,
		BGRA8,
		R8,
		RGB16F,
		RGBA16F,
		RGB32F,
		RGBA32F,
		R16F,
		R32F,
		DEPTH16,
		DEPTH24,
		DEPTH32,
		R8I,
		R8UI,
		R16I,
		R16UI,
		R32I,
		R32UI,
		RGBA8I,
		RGBA8UI,
		RGBA16I,
		RGBA16UI,
		RGBA32I,
		RGBA32UI,
		PIXEL_FORMAT_COUNT,
	};

	struct pixel_format_info
	{
		datatype type;
		uint32   elements;
	};

	inline const pixel_format_info& get_pixel_format_info( pixel_format dt )
	{
		static pixel_format_info info[PIXEL_FORMAT_COUNT] = {
			{ UBYTE, 3 }, // RGB8,
			{ UBYTE, 4 }, // RGBA8,
			{ UBYTE, 3 }, // BGR8  
			{ UBYTE, 4 }, // BGRA8,
			{ UBYTE, 1 }, // R8,
			{ FLOAT, 3 }, // RGB16F, 
			{ FLOAT, 4 }, // RGBA16F, // HALF-FLOAT
			{ FLOAT, 3 }, // RGB32F,
			{ FLOAT, 4 }, // RGBA32F,
			{ FLOAT, 1 }, // R16F  // HALF-FLOAT
			{ FLOAT, 1 }, // R32F,
			{ USHORT,1 }, // DEPTH16,
			{ UINT,  1 }, // DEPTH24,
			{ UINT,  1 }, // DEPTH32,
			{ BYTE,  1 }, // R8I,
			{ UBYTE, 1 }, // R8UI,
			{ SHORT, 1 }, // R16I,
			{ USHORT,1 }, // R16UI,
			{ INT,   1 }, // R32I,
			{ UINT,  1 }, // R32UI,
			{ BYTE,  4 }, // RGBA8I,
			{ UBYTE, 4 }, // RGBA8UI,
			{ SHORT, 4 }, // RGBA16I,
			{ USHORT,4 }, // RGBA16UI,
			{ INT,   4 }, // RGBA32I,
			{ UINT,  4 }, // RGBA32UI,
		};
		return info[dt];
	}

	class image_data
	{
	public:
		image_data( pixel_format format, ivec2 size, const uint8 * data ) 
			: m_format( format ), m_size( size ), m_data( nullptr )
		{
			NV_ASSERT_ALWAYS( data, "Data = nullptr!" );
			initialize( data );
		}
		image_data( pixel_format format, ivec2 size )
			: m_format( format ), m_size( size ), m_data( nullptr )
		{
			initialize( nullptr );
		}
		void fill( uint8 value = 0 )
		{
			uint32 bsize = static_cast<uint32>( m_size.x * m_size.y ) * get_depth();
			raw_fill_n( m_data, bsize, value );
		}
		uint8* release_data() { uint8* r = m_data; m_data = nullptr; return r; }
		const uint8 * get_data()    const { return m_data; }
		const ivec2 get_size() const { return m_size; }
		uint32 get_depth() const
		{
			return get_pixel_format_info( m_format ).elements;
		}
		datatype get_type() const
		{
			return get_pixel_format_info( m_format ).type;
		}
		pixel_format get_format() const { return m_format; }
		~image_data() {	if (m_data) delete[] m_data; }
	private:
		void initialize( const uint8* data )
		{
			uint32 bsize = static_cast<uint32>(m_size.x * m_size.y) * get_depth();
			m_data = new uint8[ bsize ]; 
			if ( data )
				raw_copy( data, data + bsize, m_data );
		}

		pixel_format m_format;
		ivec2  m_size;  //!< Defines the size of the image as a vector
		uint8* m_data;  //!< Holder for data
	};
}

#endif // NV_INTERFACE_IMAGE_DATA_HH
