// Copyright (C) 2011-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/gfx/image.hh"
#include "nv/stl/algorithm.hh"

using namespace nv;

image::image( ivec2 size, uint32 depth )
	: m_size( size ), m_depth( depth ), m_data( nullptr )
{
	m_data = new uint8[ static_cast<uint32>( m_size.x * m_size.y ) * m_depth ];
}

image::image( image_data* data )
	: m_size( data->get_size() ), m_depth( data->get_depth() ), m_data( data->release_data() )
{
	NV_ASSERT( m_data, "image created from empty image_data!" );
}


image::image( ivec2 size, uint32 depth, const uint8 * data, bool reversed )
	: m_size( size ), m_depth( depth ), m_data( nullptr )
{
	NV_ASSERT( size.x >= 0 && size.y >= 0, "bad parameters passed to image!" );
	sint32 bsize = m_size.x * m_size.y * static_cast<sint32>( m_depth );
	m_data = new uint8[ uint32( bsize ) ];

	if ( reversed )
	{
		sint32 bline = m_size.x * static_cast<sint32>( m_depth );
		for( int i = 0; i < m_size.y; ++i )
		{
			raw_copy( data + i * bline, data + (i + 1) * bline, m_data + bsize - ( i + 1 ) * bline );
		}

	}
	else 
	{
		raw_copy( data, data + bsize, m_data );
	}
}

void image::fill( uint8 value )
{
	raw_fill( m_data, m_data + m_size.x * m_size.y * static_cast<int>( m_depth ), value );
}

void image::fill( region r, uint8 value, int stride )
{
	if ( stride == 0 ) stride = r.size.x * static_cast<sint32>( m_depth );

	sint32 bpos  = (r.pos.y*m_size.x + r.pos.x ) * static_cast<sint32>( m_depth );
	sint32 bline = m_size.x*static_cast<sint32>( m_depth );

	for( int i = 0; i < r.size.y; ++i )
	{
		// TODO: test 
		raw_fill( m_data + bpos + bline * i, m_data + bpos + bline * i + stride, value );
	}
}


void image::set_region( region r, const uint8 * data, int stride )
{
	if ( stride == 0 ) stride = r.size.x * static_cast<sint32>( m_depth );
	
	sint32 bpos  = (r.pos.y*m_size.x + r.pos.x ) * static_cast<sint32>( m_depth );
	sint32 bline = m_size.x*static_cast<sint32>( m_depth );

	for( int i = 0; i < r.size.y; ++i )
 	{
// TODO: test if same as old:
//		memcpy( m_data+((r.pos.y+i)*m_size.x + r.pos.x ) * m_depth, 
//			data + (i*stride), r.size.x * m_depth );
		raw_copy( data + i*stride, data + (i+1)*stride, m_data + bpos + bline * i );
	}
}

void image::set_region( region r, const image_data* idata )
{
	if ( idata->get_depth() == m_depth )
	{
		set_region( r, idata->get_data() );
		return;
	}

	fill( r, 255 );

	uint32 bpos       = static_cast< uint32 >( r.pos.y*m_size.x + r.pos.x ) * m_depth;
	uint32 bline      = static_cast< uint32 >( m_size.x ) * m_depth;

	uint32 rsizex     = static_cast< uint32 >( r.size.x );
	uint32 rsizey     = static_cast< uint32 >( r.size.y );
	const uint8* data = idata->get_data();
	uint32 depth      = idata->get_depth();
	uint32 cdepth     = m_depth > depth ? depth : m_depth;
	uint32 dstride    = rsizex * depth;

	for( uint32 y = 0; y < rsizey; ++y )
	{
		uint32 pos = bpos + bline * y;
		for( uint32 x = 0; x < rsizex; ++x )
		{
			uint32 xy = pos + x * m_depth;
			for( uint32 e = 0; e < cdepth; ++e )
			{
				m_data[ xy + e ] = data[ y*dstride + x * depth + e ];
			}
		}
	}
}


image::~image()
{
	delete[] m_data;
}


