// Copyright (C) 2015-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_STL_STRING_STRING_TWINE_HH
#define NV_STL_STRING_STRING_TWINE_HH

#include <nv/stl/string/common.hh>
#include <nv/stl/string/string_base.hh>

namespace nv
{

	class string_twine
	{
	private:
		enum twine_type : uint8
		{
			EMPTY,
			TWINE,
			STRING,
			CSTRING,
			BUFFER,
		};
	public:

		string_twine() {}

		string_twine( uint32 u )
		{
			m_lhs.type = BUFFER;
			m_lhs.value.buffer.size = static_cast< uint8 >( uint32_to_buffer( array_ref<char>( m_lhs.value.buffer.data ), u ) );
		}

		string_twine( sint32 s )
		{
			m_lhs.type = BUFFER;
			m_lhs.value.buffer.size = static_cast< uint8 >( sint32_to_buffer( array_ref<char>( m_lhs.value.buffer.data ), s ) );
		}

		string_twine( uint64 u )
		{
			m_lhs.type = BUFFER;
			m_lhs.value.buffer.size = static_cast< uint8 >( uint64_to_buffer( array_ref<char>( m_lhs.value.buffer.data ), u ) );
		}

		string_twine( sint64 s )
		{
			m_lhs.type = BUFFER;
			m_lhs.value.buffer.size = static_cast< uint8 >( sint64_to_buffer( array_ref<char>( m_lhs.value.buffer.data ), s ) );
		}

		string_twine( f32 f )
		{
			m_lhs.type = BUFFER;
			m_lhs.value.buffer.size = static_cast< uint8 >( f64_to_buffer( array_ref<char>( m_lhs.value.buffer.data ), f ) );
		}

		string_twine( f64 f )
		{
			m_lhs.type = BUFFER;
			m_lhs.value.buffer.size = static_cast< uint8 >( f64_to_buffer( array_ref<char>( m_lhs.value.buffer.data ), f ) );
		}

		template < typename Base >
		string_twine( const string_base< Base >& s )
		{
			m_lhs.type = CSTRING;
			m_lhs.value.cstring.size = s.size();
			m_lhs.value.cstring.data = s.data();
		}

		string_twine( const string_twine& s ) = default;

		string_twine( const char* data, uint32 sz )
			: m_lhs( data, sz ), m_rhs()
		{
		}

		bool empty() const
		{
			return m_lhs.type == EMPTY;
		}

		bool is_unary() const
		{
			return m_lhs.type != EMPTY && m_rhs.type == EMPTY;
		}

		bool is_binary() const
		{
			return m_lhs.type != EMPTY && m_rhs.type != EMPTY;
		}

		string_twine concat( const string_twine& other ) const
		{
			if ( empty() ) return other;
			if ( other.empty() ) return *this;
			return string_twine( *this, other );
		}

		uint32 dump_size() const
		{
			return dump_size( m_lhs ) + dump_size( m_rhs );
		}

		void dump( array_ref< char > target ) const
		{
			dump_recursive( target );
		}
	private:
		struct twine_cstring
		{
			uint32      size;
			const char* data;
		};

		struct twine_buffer
		{
			uint8 size;
			char  data[15];
		};

		union twine_value
		{
			const string_twine* twine;
			const string_view*  string;
			twine_cstring       cstring;
			twine_buffer        buffer;
		};

		struct twine_node
		{
			twine_type type;
			twine_value value;

			twine_node() : type( EMPTY ) {}
			twine_node( twine_type t ) : type( t ) {}
			twine_node( const string_twine* v ) : type( TWINE ) { value.twine = v; }
			twine_node( const string_view* v ) : type( STRING ) { value.string = v; }
			twine_node( const char* d, uint32 s ) : type( CSTRING ) { value.cstring.data = d; value.cstring.size = s; }
		};

		twine_node m_lhs;
		twine_node m_rhs;

	private:
		string_twine &operator=( const string_twine& ) = delete;

		explicit string_twine( const string_twine& lhs, const string_twine& rhs )
		{
			if ( lhs.empty() )
			{
				m_lhs = rhs.m_lhs;
				m_rhs = rhs.m_rhs;
				return;
			}
			if ( rhs.empty() )
			{
				m_lhs = lhs.m_lhs;
				m_rhs = lhs.m_rhs;
				return;
			}
			m_lhs = lhs.is_unary() ? lhs.m_lhs : twine_node( &lhs );
			m_rhs = rhs.is_unary() ? rhs.m_lhs : twine_node( &rhs );
		}

		uint32 dump_size( const twine_node& node ) const
		{
			switch ( node.type )
			{
			case TWINE      : return node.value.twine->dump_size(); 
			case STRING     : return node.value.string->size(); 
			case CSTRING    : return node.value.cstring.size;
			case BUFFER     : return node.value.buffer.size;
			default : break;
			}
			return 0;
		}

		void dump_recursive( array_ref< char >& target ) const
		{
			dump_node( target, m_lhs );
			dump_node( target, m_rhs );
		}

		void dump_node( array_ref< char >& target, const twine_node& node ) const
		{
			switch ( node.type )
			{
			case TWINE   : node.value.twine->dump_recursive( target ); return;
			case STRING  : dump_string( target, *node.value.string ); return;
			case CSTRING : dump_string( target, string_view( node.value.cstring.data, node.value.cstring.size ) ); return;
			case BUFFER  : dump_string( target, string_view( node.value.buffer.data, node.value.buffer.size ) ); return;
			default: return;
			}
		}

		void dump_string( array_ref< char >& target, const string_view& string ) const
		{
			uint32 dump_length = nv::min( target.size(), string.size() );
			if ( dump_length > 0 )
			{
				raw_copy_n( string.data(), dump_length, target.data() );
				target.assign( target.data() + dump_length, target.size() - dump_length );
			}
		}

	};

	inline string_twine operator+( const string_twine& lhs, const string_twine& rhs )
	{
		return lhs.concat( rhs );
	}

	template < size_t N >
	inline string_twine operator+( const char (&lhs)[N], const string_twine& rhs )
	{
		return string_twine( lhs, static_cast< uint32 >( N-1 ) ).concat( rhs );
	}

	template < size_t N >
	inline string_twine operator+( const string_twine& lhs, const char( &rhs )[N] )
	{
		return lhs.concat( string_twine( rhs, static_cast< uint32 >( N - 1 ) ) );
	}

}

#endif // NV_STL_STRING_STRING_TWINE_HH
