// Copyright (C) 2017-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file component_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_COMPONENT_HH
#define NV_ECS_COMPONENT_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/priority_queue.hh>
#include <nv/stl/range.hh>
#include <nv/core/types.hh>
#include <nv/ecs/component_storage.hh>

namespace nv
{

	namespace ecs
	{

		template < typename Ecs, typename Component >
		class component : public Ecs::component_interface
		{
		public:
			typedef Ecs                                      ecs_type;
			typedef typename ecs_type::message               message_type;
			typedef typename ecs_type::handle_type           handle_type;
			typedef typename ecs_type::time_type             time_type;
			typedef Component                                value_type;
			typedef Component                                component_type;
			typedef index_storage< handle_type, value_type > storage_type;
			typedef typename storage_type::index_type        index_type;

			typedef typename storage_type::iterator          iterator;
			typedef typename storage_type::const_iterator    const_iterator;
			typedef typename storage_type::reference         reference;
			typedef typename storage_type::const_reference   const_reference;

			component( ecs_type& a_ecs, string_view a_name, uint32 reserve = 0 ) 
				: m_ecs( a_ecs ), m_data( reserve )
			{
				m_ecs.register_component<component_type>( a_name, this );
			}

			inline value_type& insert( handle_type h ) { return m_data.insert( h ); }

			template < typename ...Args >
			value_type& insert( handle_type h, Args&&... args )
			{
				return m_data.insert( h, nv::forward<Args>( args )... );
			}

			bool exists( handle_type h )
			{
				return m_data.exists( h );
			}

			virtual void update( time_type /*dtime*/ )
			{
				// no-op
			}

			virtual void clear()
			{
				for ( uint32 i = 0; i < size(); ++i )
					destroy( &m_data[i] );
				m_data.clear();
			}

			value_type* get( handle_type h ) { return m_data.get(h); }
			const value_type* get( handle_type h ) const { return m_data.get( h ); }

			void* get_raw( handle_type h ) { return get( h ); }
			const void* get_raw( handle_type h ) const { return get( h ); }

			virtual void remove( handle_type h )
			{
				value_type* v = m_data.get( h );
				if ( v == nullptr ) return;
				destroy( v );
				m_data.remove( h );
			}

			virtual void destroy( value_type* )
			{
				// cleanup
			}

			virtual bool handle_message( const message_type& )
			{
				return true;
			}

			~component()
			{
				clear();
			}


			inline handle_type get_handle( index_type i ) const { return m_data.get_handle( i ); }

			inline const value_type& operator[] ( index_type i ) const { return m_data[i]; }
			inline value_type& operator[] ( index_type i ) { return m_data[i]; }

			inline size_t size() const { return m_data.size(); }

			inline iterator        begin() { return m_data.begin(); }
			inline const_iterator  begin()  const { return m_data.begin(); }
			inline const_iterator  cbegin() const { return m_data.begin(); }

			inline iterator        end() { return m_data.end(); }
			inline const_iterator  end()  const { return m_data.end(); }
			inline const_iterator  cend() const { return m_data.end(); }
		protected:
			ecs_type&        m_ecs;
			storage_type     m_data;
		};

	}

}

#endif // NV_ECS_COMPONENT_HH
