// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file logging.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief logging interfaces
 */

#ifndef NV_CORE_LOGGING_HH
#define NV_CORE_LOGGING_HH

#include <nv/common.hh>
#include <nv/base/capi.hh>
#include <nv/stl/string.hh>
#include <nv/stl/singleton.hh>

namespace nv
{

	class logger_base : public singleton< logger_base >
	{
	public:
		logger_base()
		{
			m_pos = m_message;
		}
		unsigned int get_level()
		{
			return m_level;
		}
		void set_level( unsigned int level )
		{
			m_level = level;
		}
		virtual void log( log_level level, const string_view& message ) = 0;
		void log_append( string_view ref )
		{
			uint32 count = ::nv::min< uint32 >( ref.size(), space_left()-1 );
			nvmemcpy( m_pos, ref.data(), count );
			m_pos += count;
		}
		void log_append( uint32 u )
		{
			m_pos += uint32_to_buffer( array_ref< char >( m_pos, space_left() ), u );
		}
		void log_append( sint32 s )
		{
			m_pos += sint32_to_buffer( array_ref< char >( m_pos, space_left() ), s );
		}
		void log_append( uint64 u )
		{
			m_pos += uint64_to_buffer( array_ref< char >( m_pos, space_left() ), u );
		}
		void log_append( sint64 s )
		{
			m_pos += sint64_to_buffer( array_ref< char >( m_pos, space_left() ), s );
		}
		void log_append( f32 f )
		{
			m_pos += f32_to_buffer( array_ref< char >( m_pos, space_left() ), f );
		}
		static bool can_log( log_level level )
		{
			return logger_base::is_valid() && static_cast<unsigned int>( reference().get_level() ) >= static_cast<unsigned int>( level );
		}

		void log_sequence( log_level level )
		{
			*m_pos = '\0';
			log( level, string_view( m_message, static_cast<uint32>( m_pos - m_message ) ) );
			m_pos = m_message;
		}
		template < typename T, typename... Args >
		inline void log_sequence( log_level level, T&& t, Args&&... args )
		{
			this->log_append( t );
			this->log_sequence( level, nv::forward<Args>( args )... );
		}
		virtual ~logger_base() {}
	protected:
		inline uint32 space_left() const
		{
			return 1024 - static_cast< uint32 >( m_pos - m_message );
		}

		char m_message[1024];
		char* m_pos;
		unsigned int m_level;
	};

} // namespace nv

#define NV_LOG(level, ...) \
	if ( nv::logger_base::can_log(level) ) \
	{\
		nv::logger_base::reference().log_sequence( level, __VA_ARGS__ ); \
	}

#if NV_DEBUG == 1
#define NV_DEBUG_LOG(level, ...) \
	if ( nv::logger_base::can_log(level) ) \
	{\
		nv::logger_base::reference().log_sequence( level, __VA_ARGS__ ); \
	}
#else
#define NV_DEBUG_LOG(level, ...)
#endif

#define NV_LOG_FATAL(...)    NV_LOG( nv::LOG_FATAL, __VA_ARGS__ )
#define NV_LOG_CRITICAL(...) NV_LOG( nv::LOG_CRITICAL, __VA_ARGS__ )
#define NV_LOG_ERROR(...)    NV_LOG( nv::LOG_ERROR, __VA_ARGS__ )
#define NV_LOG_WARNING(...)  NV_LOG( nv::LOG_WARNING, __VA_ARGS__ )
#define NV_LOG_NOTICE(...)   NV_LOG( nv::LOG_NOTICE, __VA_ARGS__ )
#define NV_LOG_INFO(...)     NV_LOG( nv::LOG_INFO, __VA_ARGS__ )
#define NV_LOG_DEBUG(...)    NV_LOG( nv::LOG_DEBUG, __VA_ARGS__ )
#define NV_LOG_TRACE(...)    NV_LOG( nv::LOG_TRACE, __VA_ARGS__ )

#endif // NV_CORE_LOGGING_HH
