// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file vec4.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief vec4
 */

#ifndef NV_STL_MATH_VEC4_HH
#define NV_STL_MATH_VEC4_HH

#include <nv/stl/math/common.hh>

namespace nv
{

	namespace math
	{

		template <typename T>
		struct tvec4
		{
			typedef tvec4<T> type;
			typedef T value_type;
			static constexpr uint32 SIZE = 4;

			union
			{
				struct { T x, y, z, w; };
				struct { T r, g, b, a; };
				struct { T s, t, p, q; };
			};

			typedef uint32 size_type;
			inline constexpr size_type size() const { return 4; }

			inline T & operator[]( size_type i )
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return ( &x )[i];
			}

			inline T operator[]( size_type i ) const
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return ( &x )[i];
			}

			inline tvec4()
				: x( static_cast<T>( 0 ) )
				, y( static_cast<T>( 0 ) )
				, z( static_cast<T>( 0 ) )
				, w( static_cast<T>( 0 ) )
			{}

			inline tvec4( const tvec4<T> & m ) = default;
			inline tvec4( tvec4<T> && m ) = default;
			inline tvec4<T> & operator=( const tvec4<T> & m ) = default;
			inline tvec4<T> & operator=( tvec4<T> && m ) = default;

			inline explicit tvec4( T scalar )
				: x( scalar )
				, y( scalar )
				, z( scalar )
				, w( scalar )
			{}

			inline explicit tvec4( no_init_t ) {}

			inline tvec4( T sx, T sy, T sz, T sw )
				: x( sx )
				, y( sy )
				, z( sz )
				, w( sw )
			{}

			template <typename A, typename B, typename C, typename D>
			inline tvec4( A sx, B sy, C sz, D sw )
				: x( static_cast<T>( sx ) )
				, y( static_cast<T>( sy ) )
				, z( static_cast<T>( sz ) )
				, w( static_cast<T>( sw ) )
			{}

			template <typename A, typename B, typename C>
			inline explicit tvec4( const tvec2<A> & a, B b, C c )
				: x( static_cast<T>( a.x ) )
				, y( static_cast<T>( a.y ) )
				, z( static_cast<T>( b ) )
				, w( static_cast<T>( c ) )
			{}

			template <typename A, typename B, typename C>
			inline explicit tvec4( A a, const tvec2<B> & b, C c )
				: x( static_cast<T>( a ) )
				, y( static_cast<T>( b.x ) )
				, z( static_cast<T>( b.y ) )
				, w( static_cast<T>( c ) )
			{}

			template <typename A, typename B, typename C>
			inline explicit tvec4( A a, B b, const tvec2<C> & c )
				: x( static_cast<T>( a ) )
				, y( static_cast<T>( b ) )
				, z( static_cast<T>( c.x ) )
				, w( static_cast<T>( c.y ) )
			{}

			template <typename A, typename B>
			inline explicit tvec4( const tvec3<A> & a, B b )
				: x( static_cast<T>( a.x ) )
				, y( static_cast<T>( a.y ) )
				, z( static_cast<T>( a.z ) )
				, w( static_cast<T>( b ) )
			{}

			template <typename A, typename B>
			inline explicit tvec4( A a, const tvec3<B> & b )
				: x( static_cast<T>( a ) )
				, y( static_cast<T>( b.x ) )
				, z( static_cast<T>( b.y ) )
				, w( static_cast<T>( b.z ) )
			{}

			template <typename A, typename B>
			inline explicit tvec4( const tvec2<A> & a, const tvec2<B> & b )
				: x( static_cast<T>( a.x ) )
				, y( static_cast<T>( a.y ) )
				, z( static_cast<T>( b.x ) )
				, w( static_cast<T>( b.y ) )
			{}

			template <typename U>
			inline explicit tvec4( const tvec4<U> & v )
				: x( static_cast<T>( v.x ) )
				, y( static_cast<T>( v.y ) )
				, z( static_cast<T>( v.z ) )
				, w( static_cast<T>( v.w ) )
			{}

			inline tvec4<T> & operator+=( T scalar )
			{
				this->x += static_cast<T>( scalar );
				this->y += static_cast<T>( scalar );
				this->z += static_cast<T>( scalar );
				this->w += static_cast<T>( scalar );
				return *this;
			}

			inline tvec4<T> & operator+=( const tvec4<T> & v )
			{
				this->x += static_cast<T>( v.x );
				this->y += static_cast<T>( v.y );
				this->z += static_cast<T>( v.z );
				this->w += static_cast<T>( v.w );
				return *this;
			}

			inline tvec4<T> & operator-=( T scalar )
			{
				this->x -= static_cast<T>( scalar );
				this->y -= static_cast<T>( scalar );
				this->z -= static_cast<T>( scalar );
				this->w -= static_cast<T>( scalar );
				return *this;
			}

			inline tvec4<T> & operator-=( const tvec4<T> & v )
			{
				this->x -= static_cast<T>( v.x );
				this->y -= static_cast<T>( v.y );
				this->z -= static_cast<T>( v.z );
				this->w -= static_cast<T>( v.w );
				return *this;
			}

			inline tvec4<T> & operator*=( T v )
			{
				this->x *= static_cast<T>( v );
				this->y *= static_cast<T>( v );
				this->z *= static_cast<T>( v );
				this->w *= static_cast<T>( v );
				return *this;
			}

			inline tvec4<T> & operator*=( const tvec4<T> & v )
			{
				this->x *= static_cast<T>( v.x );
				this->y *= static_cast<T>( v.y );
				this->z *= static_cast<T>( v.z );
				this->w *= static_cast<T>( v.w );
				return *this;
			}

			inline tvec4<T> & operator/=( T v )
			{
				this->x /= static_cast<T>( v );
				this->y /= static_cast<T>( v );
				this->z /= static_cast<T>( v );
				this->w /= static_cast<T>( v );
				return *this;
			}

			inline tvec4<T> & operator/=( const tvec4<T> & v )
			{
				this->x /= static_cast<T>( v.x );
				this->y /= static_cast<T>( v.y );
				this->z /= static_cast<T>( v.z );
				this->w /= static_cast<T>( v.w );
				return *this;
			}

			inline tvec4<T> & operator++()
			{
				++this->x;
				++this->y;
				++this->z;
				++this->w;
				return *this;
			}

			inline tvec4<T> & operator--()
			{
				--this->x;
				--this->y;
				--this->z;
				--this->w;
				return *this;
			}

			inline tvec4<T> operator++( int )
			{
				tvec4<T> result( *this );
				++*this;
				return result;
			}

			inline tvec4<T> operator--( int )
			{
				tvec4<T> result( *this );
				--*this;
				return result;
			}

		};

		template <typename T>
		inline tvec4<T> operator-( const tvec4<T> & v )
		{
			return tvec4<T>(
				-v.x,
				-v.y,
				-v.z,
				-v.w );
		}

		template <typename T>
		inline tvec4<T> operator+( const tvec4<T> & v, T scalar )
		{
			return tvec4<T>(
				v.x + scalar,
				v.y + scalar,
				v.z + scalar,
				v.w + scalar );
		}

		template <typename T>
		inline tvec4<T> operator+( T scalar, const tvec4<T> & v )
		{
			return tvec4<T>(
				scalar + v.x,
				scalar + v.y,
				scalar + v.z,
				scalar + v.w );
		}

		template <typename T>
		inline tvec4<T> operator+( const tvec4<T> & v1, const tvec4<T> & v2 )
		{
			return tvec4<T>(
				v1.x + v2.x,
				v1.y + v2.y,
				v1.z + v2.z,
				v1.w + v2.w );
		}

		template <typename T>
		inline tvec4<T> operator-( const tvec4<T> & v, T scalar )
		{
			return tvec4<T>(
				v.x - scalar,
				v.y - scalar,
				v.z - scalar,
				v.w - scalar );
		}

		template <typename T>
		inline tvec4<T> operator-( T scalar, const tvec4<T> & v )
		{
			return tvec4<T>(
				scalar - v.x,
				scalar - v.y,
				scalar - v.z,
				scalar - v.w );
		}

		template <typename T>
		inline tvec4<T> operator-( const tvec4<T> & v1, const tvec4<T> & v2 )
		{
			return tvec4<T>(
				v1.x - v2.x,
				v1.y - v2.y,
				v1.z - v2.z,
				v1.w - v2.w );
		}

		template <typename T>
		inline tvec4<T> operator*( const tvec4<T> & v, T scalar )
		{
			return tvec4<T>(
				v.x * scalar,
				v.y * scalar,
				v.z * scalar,
				v.w * scalar );
		}

		template <typename T>
		inline tvec4<T> operator*( T scalar, const tvec4<T> & v )
		{
			return tvec4<T>(
				scalar * v.x,
				scalar * v.y,
				scalar * v.z,
				scalar * v.w );
		}

		template <typename T>
		inline tvec4<T> operator*( const tvec4<T> & v1, const tvec4<T> & v2 )
		{
			return tvec4<T>(
				v1.x * v2.x,
				v1.y * v2.y,
				v1.z * v2.z,
				v1.w * v2.w );
		}

		template <typename T>
		inline tvec4<T> operator/( const tvec4<T> & v, T scalar )
		{
			return tvec4<T>(
				v.x / scalar,
				v.y / scalar,
				v.z / scalar,
				v.w / scalar );
		}

		template <typename T>
		inline tvec4<T> operator/( T scalar, const tvec4<T> & v )
		{
			return tvec4<T>(
				scalar / v.x,
				scalar / v.y,
				scalar / v.z,
				scalar / v.w );
		}

		template <typename T>
		inline tvec4<T> operator/( const tvec4<T> & v1, const tvec4<T> & v2 )
		{
			return tvec4<T>(
				v1.x / v2.x,
				v1.y / v2.y,
				v1.z / v2.z,
				v1.w / v2.w );
		}

		template <typename T>
		inline tvec4<T> operator%( const tvec4<T> & v, T scalar )
		{
			return tvec4<T>(
				v.x % scalar,
				v.y % scalar,
				v.z % scalar,
				v.w % scalar );
		}

		template <typename T>
		inline tvec4<T> operator%( T scalar, const tvec4<T> & v )
		{
			return tvec4<T>(
				scalar % v.x,
				scalar % v.y,
				scalar % v.z,
				scalar % v.w );
		}

		template <typename T>
		inline tvec4<T> operator%( const tvec4<T> & v1, const tvec4<T> & v2 )
		{
			return tvec4<T>(
				v1.x % v2.x,
				v1.y % v2.y,
				v1.z % v2.z,
				v1.w % v2.w );
		}

		template <typename T>
		inline bool operator==( const tvec4<T> & v1, const tvec4<T> & v2 )
		{
			return ( v1.x == v2.x ) && ( v1.y == v2.y ) && ( v1.z == v2.z ) && ( v1.w == v2.w );
		}

		template <typename T>
		inline bool operator!=( const tvec4<T> & v1, const tvec4<T> & v2 )
		{
			return ( v1.x != v2.x ) || ( v1.y != v2.y ) || ( v1.z != v2.z ) || ( v1.w != v2.w );
		}

	}

}

#endif // NV_STL_MATH_VEC4_HH
