// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_GFX_MESH_CREATOR_HH
#define NV_GFX_MESH_CREATOR_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/core/aabb.hh>
#include <nv/interface/mesh_data.hh>

namespace nv
{

	class mesh_data_creator
	{
	public:
		mesh_data_creator( data_channel_set* data ) : m_data( data )
		{
			initialize();
		}

		aabb calculate_aabb();

		// assumes that position and normal is vec3, tangent is vec4
		void transform( const vec3& pos, const mat3& r33, float scale = 1.0f );
		// assumes that position and normal is vec3, tangent is vec4
		void transform( const vec3& pos, const quat& r, float scale = 1.0f )
		{
			transform( pos, mat3_cast(r), scale );
		}
		// assumes that position and normal is vec3, tangent is vec4
		void transform( const nv::transform& tr, float scale = 1.0f )
		{
			transform( tr.get_position(), tr.get_orientation(), scale );
		}
		// assumes that position and normal is vec3, tangent is vec4
		void transform( float scale, const mat3& r33 )
		{
			// remove this
			transform( vec3(), r33, scale );
		}
		// TODO: this could generate normals too
		void generate_tangents();
		// assumes that position and normal is vec3, tangent is vec4
		void rotate_quadrant( uint8 rotation );
		// assumes that position and normal is vec3, tangent is vec4
		void mirror( bool x, bool z );
		// assumes mesh is indexed
		void swap_culling();
		void translate( vec3 offset );
		void flip_normals();
		void scale_texture( vec2 min, vec2 max );
		bool is_same_format( const data_channel_set* other );
		void merge( const data_channel_set* other );
	private:
		void initialize();

		raw_data_channel merge_channels( const raw_data_channel& a, const raw_data_channel& b );
		raw_data_channel append_channels( const raw_data_channel& a, const raw_data_channel& b, uint32 frame_count = 1 );

		raw_data_channel* m_pos_channel;
		raw_data_channel* m_nrm_channel;
		raw_data_channel* m_tan_channel;
		raw_data_channel* m_tex_channel;
		raw_data_channel* m_idx_channel;

		int m_pos_offset;
		int m_nrm_offset;
		int m_tan_offset;
		int m_tex_offset;
		int m_idx_offset;

		nv::datatype m_pos_type;
		nv::datatype m_nrm_type;
		nv::datatype m_tan_type;
		nv::datatype m_tex_type;
		nv::datatype m_idx_type;

		data_channel_set* m_data;
	};

	class mesh_nodes_creator
	{
	public:
		mesh_nodes_creator( mesh_nodes_data* data ) : m_data( data )
		{}
		// assumes that keys are equally spaced
		void merge_keys();
		void transform( float scale, const mat3& r33 );
	private:
		mesh_nodes_data* m_data;
	};

	class data_node_list_creator
	{
	public:
		data_node_list_creator( data_node_list* data ) : m_data( data ) {}
		void transform( float scale, const mat3& r33 );
	private:
		data_node_list* m_data;
	};


}

#endif // NV_GFX_MESH_CREATOR_HH
