// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file mesh_manager.hh
 * @author Kornel Kisielewicz
 * @brief mesh_manager
 */

#ifndef NV_ENGINE_MESH_MANAGER_HH
#define NV_ENGINE_MESH_MANAGER_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/gfx/skeleton_instance.hh>
#include <nv/engine/resource_system.hh>


namespace nv
{

	enum shader_geo
	{
		DIRECT = 0,
		BONE = 1,
	};

	struct gpu_mesh
	{
		vertex_array va;
		uint32       count;
		shader_geo  shader;
	};

	struct mesh_data
	{
		string128                              path;
		vector< data_node_info >               infos;
		vector< resource< data_channel_set > > meshes;
		hash_store< shash64, uint32 >          names;
		hash_store< shash64, uint32 >          node_names;
		data_node_list*                        node_data;
	};

	NV_RTTI_DECLARE_NAME( data_channel_set, "data_channel_set" )
	NV_RTTI_DECLARE_NAME( gpu_mesh, "gpu_mesh" )
	NV_RTTI_DECLARE_NAME( mesh_data, "mesh_data" )

	using mesh_manager = manual_resource_manager< data_channel_set >;

	class gpu_mesh_manager : public dependant_resource_manager< gpu_mesh, mesh_manager >
	{
	public:
		gpu_mesh_manager( context* context, mesh_manager* mesh_manager )
			: dependant_resource_manager( mesh_manager )
			, m_context( context ) {}
	protected:
		resource< gpu_mesh > create_resource( resource< data_channel_set > mesh );
		virtual void release( gpu_mesh* m );
	private:
		context*      m_context;
	};

	struct mesh_source
	{
		resource< mesh_data > mesh;
		uint32                index;
	};

	class mesh_data_manager : public file_resource_manager< mesh_data >
	{
	public:
		mesh_data_manager( mesh_manager* meshes )
			: m_mesh_manager( meshes ) , m_strings( new string_table ) {}
		string_view resolve( shash64 h ) { return ( *m_strings )[h]; }
		resource< data_channel_set > get_path( const string_view& path,
			resource< mesh_data > default = resource< mesh_data >(),
			data_node_info* info = nullptr );

		mesh_source get_source( resource< data_channel_set > mesh )
		{
			auto it = m_source_map.find( mesh.id() );
			if ( it != m_source_map.end() ) return it->second;
			return mesh_source{};
		}

		~mesh_data_manager() { delete m_strings; }
	protected:
		virtual bool load_resource( const string_view& id );
		nv::hash_store< resource_id, mesh_source > m_source_map;
		mesh_manager*                              m_mesh_manager;
		string_table*                              m_strings;
	};

}

#endif // NV_ENGINE_MESH_MANAGER_HH
