// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file random_access.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief random_access classes
*/

#ifndef NV_STL_CONTAINER_RANDOM_ACCESS_HH
#define NV_STL_CONTAINER_RANDOM_ACCESS_HH

#include <nv/common.hh>
#include <nv/stl/algorithm/fill.hh>
#include <nv/stl/iterator.hh>


namespace nv
{
	template < typename Super, bool Const = Super::is_const >
	class random_access {};

	template < typename Super >
	class random_access < Super, true > : public Super
	{
	public:
		typedef typename Super::value_type           value_type;
		typedef typename Super::size_type            size_type;
		typedef ptrdiff_t                            difference_type;
		typedef const value_type*                    pointer;
		typedef const value_type*                    const_pointer;
		typedef const value_type*                    iterator;
		typedef const value_type*                    const_iterator;
		typedef const value_type*                    reference;
		typedef const value_type*                    const_reference;
		typedef nv::reverse_iterator<iterator>       reverse_iterator;
		typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

		using Super::Super;

		inline const_reference front() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[0]; }
		inline const_reference back() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[Super::size() - 1]; }

		const_reference operator[]( size_type i ) const
		{
			NV_ASSERT( i < Super::size(), "Out of range" );
			return Super::data()[i];
		}

		inline const_iterator begin() const { return const_iterator( Super::data() ); }
		inline const_iterator end() const { return const_iterator( Super::data() + Super::size() ); }
		inline const_iterator cbegin() const { return const_iterator( Super::data() ); }
		inline const_iterator cend() const { return const_iterator( Super::data() + Super::size() ); }
		inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
		inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
		inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
		inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
		inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= Super::size(), "Index out of range" ); return begin() + i; }
	};

	template < typename Super >
	class random_access < Super, false > : public Super
	{
	public:
		typedef typename Super::value_type           value_type;
		typedef typename Super::size_type            size_type;
		typedef ptrdiff_t                            difference_type;
		typedef value_type*                          pointer;
		typedef const value_type*                    const_pointer;
		typedef value_type*					         iterator;
		typedef const value_type*                    const_iterator;
		typedef value_type&                          reference;
		typedef const value_type&                    const_reference;
		typedef nv::reverse_iterator<iterator>       reverse_iterator;
		typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

		using Super::Super;

		inline reference       front() { NV_ASSERT( !Super::empty(), "front() called on empty data!" );  return Super::data()[0]; }
		inline const_reference front() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[0]; }
		inline reference       back() { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[Super::size() - 1]; }
		inline const_reference back() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[Super::size() - 1]; }

		reference operator[]( size_type i )
		{
			NV_ASSERT( i < Super::size(), "Out of range" );
			return Super::data()[i];
		}

		const_reference operator[]( size_type i ) const
		{
			NV_ASSERT( i < Super::size(), "Out of range" );
			return Super::data()[i];
		}

		inline void fill( const value_type& value )
		{
			::nv::fill( iterator( Super::data() ), iterator( Super::data() + this->size() ), value );
		}

		inline const_iterator begin() const { return const_iterator( Super::data() ); }
		inline const_iterator end() const { return const_iterator( Super::data() + Super::size() ); }
		inline iterator begin() { return iterator( Super::data() ); }
		inline iterator end() { return iterator( Super::data() + Super::size() ); }
		inline const_iterator cbegin() const { return const_iterator( Super::data() ); }
		inline const_iterator cend() const { return const_iterator( Super::data() + Super::size() ); }
		inline reverse_iterator rbegin() { return reverse_iterator( end() ); }
		inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
		inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
		inline reverse_iterator rend() { return reverse_iterator( begin() ); }
		inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
		inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
		inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= Super::size(), "Index out of range" ); return begin() + i; }
		inline iterator       iat( size_t i ) { NV_ASSERT( i <= Super::size(), "Index out of range" ); return begin() + i; }
	};
}

#endif // NV_STL_CONTAINER_RANDOM_ACCESS_HH
