// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_STL_STRING_SHORT_STRING_HH
#define NV_STL_STRING_SHORT_STRING_HH

#include <nv/stl/string/common.hh>
#include <nv/stl/string/string_base.hh>
#include <nv/stl/string/string_twine.hh>
#include <nv/stl/container/random_access.hh>
#include <nv/stl/container/contiguous_storage.hh>
#include <nv/stl/container/growing_storage.hh>

namespace nv
{
	template < typename Storage >
	class string_buffer_base : public string_base< Storage >
	{
	public:
		typedef string_buffer_base< Storage >               this_type;
		typedef string_base< Storage >                      base_type;
		typedef typename string_base< Storage >::size_type  size_type;
		typedef typename Storage::iterator                  iterator;
		typedef typename Storage::const_iterator            const_iterator;

		constexpr string_buffer_base() : base_type() {}
		inline string_buffer_base( const char* data, size_type sz )
			: base_type()
		{
			append( data, sz );
		}

		template < typename Base >
		inline explicit string_buffer_base( const string_base< Base >& s ) : this_type( s.data(), s.size() ) {}
		inline explicit string_buffer_base( const string_view& s ) : this_type( s.data(), s.size() ) {}

		inline string_buffer_base( const string_twine& twine )
		{
			append( twine );
		}
		inline string_buffer_base( string_buffer_base&& copy ) = default;
		inline string_buffer_base& operator=( string_buffer_base&& other ) = default;

		inline string_buffer_base( const string_buffer_base& copy )
		{
			assign( copy.data(), copy.size() );
		}

		inline string_buffer_base& operator=( const string_buffer_base& other )
		{
			assign( other.data(), other.size() );
			return ( *this );
		}


		using Storage::clear;

		void assign( const char* data, size_type sz )
		{
			Storage::clear();
			append( data, sz );
		}

		template < typename T >
		void assign( const T& value )
		{
			Storage::clear();
			append( value );
		}

		size_type append( const char* data, size_type sz )
		{
			if ( !data ) return 0;
			size_type pos    = this->size();
			size_type amount = expand_by( sz );
			raw_copy_n( data, amount, this->begin() + pos );
			return amount;
		}

		size_type append( const string_twine& twine )
		{
			size_type pos = this->size();
			size_type dump_size = twine.dump_size();
			expand_by( dump_size );
			twine.dump( array_ref< char >( this->data() + pos, this->size() - pos ) );
			return this->size() - pos;
		}

		size_type append( const_iterator first, const_iterator last )
		{
			return append( first, last - first );
		}

		size_type append( const string_view& s )
		{
			return append( s.data(), s.size() );
		}

		template < typename S >
		size_type append( const string_base<S>& s )
		{
			return append( s.data(), s.size() );
		}

		size_type append( sint32 s )
		{
			char buffer[8];
			size_type result = sint32_to_buffer( array_ref< char >( buffer ), s );
			return ( result > 0 ? append( buffer, result ) : 0 );
		}

		size_type append( uint32 u )
		{
			char buffer[8];
			size_type result = uint32_to_buffer( array_ref< char >( buffer ), u );
			return ( result > 0 ? append( buffer, result ) : 0 );
		}

		size_type append( sint64 s )
		{
			char buffer[16];
			size_type result = sint64_to_buffer( array_ref< char >( buffer ), s );
			return ( result > 0 ? append( buffer, result ) : 0 );
		}

		size_type append( uint64 u )
		{
			char buffer[16];
			size_type result = uint64_to_buffer( array_ref< char >( buffer ), u );
			return ( result > 0 ? append( buffer, result ) : 0 );
		}

		size_type append( f32 f )
		{
			char buffer[64];
			size_type result = f32_to_buffer( array_ref< char >( buffer ), f );
			return ( result > 0 ? append( buffer, result ) : 0 );
		}

		size_type append( f64 f )
		{
			char buffer[64];
			size_type result = f64_to_buffer( array_ref< char >( buffer ), f );
			return ( result > 0 ? append( buffer, result ) : 0 );
		}

		void reserve( size_type new_capacity )
		{
			Storage::reserve( new_capacity + 1 );
			this->data()[this->size()] = 0;
		}

		void resize( size_type new_size )
		{
			Storage::reserve( new_size + 1 );
			Storage::resize( min( new_size, this->capacity() - 1 ) );
			this->data()[this->size()] = 0;
		}

		iterator erase( iterator position )
		{
			iterator result = Storage::erase( position );
			this->data()[this->size()] = 0;
			return result;
		}

		iterator erase( iterator first, iterator last )
		{
			iterator result = Storage::erase( first, last );
			this->data()[this->size()] = 0;
			return result;
		}

	protected:

		size_type expand_by( size_type sz )
		{
			size_type result = 0;
			if ( Storage::try_grow( sz, 1 ) )
				result = sz;
			else
			{
				result = this->capacity() - this->size() - 1;
				Storage::try_resize( this->capacity() - 1, true, 1 );
			}
			this->data()[this->size()] = 0;
			return result;
		}
	private: // blocked, because not taking care of trailing zero
		using Storage::assign;
		using Storage::insert;
		using Storage::pop_back;
		using Storage::push_back;
		using Storage::emplace_back;
	};

	template < size_t N >
	using short_string = 
		string_buffer_base<
			random_access <
				growing_storage<
					static_storage< char, N >,
						policy_initialize_never,
							uint8, N > > >;

	using string_buffer = 
		string_buffer_base<
			random_access <
				growing_storage<
					dynamic_storage< char >,
						policy_initialize_never,
							uint16 > > >;

	using string32  = short_string< 31 >;
	using string64  = short_string< 63 >;
	using string128 = short_string< 127 >;
	using string256 = short_string< 255 >;

}

NV_RTTI_DECLARE( nv::string32 )
NV_RTTI_DECLARE( nv::string64 )
NV_RTTI_DECLARE( nv::string128 )
NV_RTTI_DECLARE( nv::string256 )

#endif // NV_STL_STRING_SHORT_STRING_HH
