// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file mat3.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief matrix 3x3
*/

#ifndef NV_STL_MATH_MAT3_HH
#define NV_STL_MATH_MAT3_HH

#include <nv/stl/math/common.hh>

namespace nv
{

	namespace math
	{
		template <typename T>
		struct tmat3
		{
			typedef tvec3<T> column_type;
			typedef tvec3<T> row_type;
			typedef tmat3<T> type;
			typedef size_t   size_type;
			typedef T        value_type;

			static constexpr size_type SIZE = 3;
			inline constexpr size_type size() const { return 3; }
		private:
			column_type value[3];

		public:

			inline tmat3()
				: value{
					column_type( static_cast<T>( 1 ), static_cast<T>( 0 ), static_cast<T>( 0 ) ),
					column_type( static_cast<T>( 0 ), static_cast<T>( 1 ), static_cast<T>( 0 ) ),
					column_type( static_cast<T>( 0 ), static_cast<T>( 0 ), static_cast<T>( 1 ) )
				}
			{}

			inline tmat3( const tmat3<T>& m ) = default;
			inline tmat3( tmat3<T>&& m ) = default;
			inline tmat3<T> & operator=( const tmat3<T> & m ) = default;
			inline tmat3<T> & operator=( tmat3<T> && m ) = default;

			inline explicit tmat3( T s )
				: value{
					column_type( static_cast<T>( s ), static_cast<T>( 0 ), static_cast<T>( 0 ) ),
					column_type( static_cast<T>( 0 ), static_cast<T>( s ), static_cast<T>( 0 ) ),
					column_type( static_cast<T>( 0 ), static_cast<T>( 0 ), static_cast<T>( s ) )
				}
			{}

			inline explicit tmat3( no_init_t )
				: value{
					column_type( no_init ),
					column_type( no_init ),
					column_type( no_init )
				}
			{}

			inline tmat3(
				T x0, T y0, T z0,
				T x1, T y1, T z1,
				T x2, T y2, T z2
			) : value{
				column_type( x0, y0, z0 ),
				column_type( x1, y1, z1 ),
				column_type( x2, y2, z2 )
			}
			{}

			inline tmat3(
				const column_type & v0,
				const column_type & v1,
				const column_type & v2
			) : value{ v0, v1, v2 }
			{}

			template <typename U>
			inline explicit tmat3( const tmat3<U> & m )
				: value{ 
					column_type( m[0] ),
					column_type( m[1] ),
					column_type( m[2] )
				}
			{}

			inline explicit tmat3( const tmat2<T> & m )
				: value{ 
					column_type( m[0], static_cast<T>( 0 ) ),
					column_type( m[1], static_cast<T>( 0 ) ),
					column_type( static_cast<T>( 0 ), static_cast<T>( 1 ) )
				}
			{}

			inline explicit tmat3( const tmat4<T> & m )
				: value{ 
					column_type( m[0] ),
					column_type( m[1] ),
					column_type( m[2] )
				}
			{}

			inline column_type & operator[]( size_type i )
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return this->value[i];
			}

			inline const column_type& operator[]( size_type i ) const
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return this->value[i];
			}


			inline tmat3<T> & operator+=( T s )
			{
				this->value[0] += s;
				this->value[1] += s;
				this->value[2] += s;
				return *this;
			}

			inline tmat3<T> & operator+=( const tmat3<T> & m )
			{
				this->value[0] += m[0];
				this->value[1] += m[1];
				this->value[2] += m[2];
				return *this;
			}

			inline tmat3<T> & operator-=( T s )
			{
				this->value[0] -= s;
				this->value[1] -= s;
				this->value[2] -= s;
				return *this;
			}

			inline tmat3<T> & operator-=( const tmat3<T> & m )
			{
				this->value[0] -= m[0];
				this->value[1] -= m[1];
				this->value[2] -= m[2];
				return *this;
			}

			inline tmat3<T> & operator*=( T s )
			{
				this->value[0] *= s;
				this->value[1] *= s;
				this->value[2] *= s;
				return *this;
			}

			inline tmat3<T> & operator*=( const tmat3<T> & m )
			{
				return ( *this = *this * m );
			}

			inline tmat3<T> & operator/=( T s )
			{
				this->value[0] /= s;
				this->value[1] /= s;
				this->value[2] /= s;
				return *this;
			}

			inline tmat3<T> & operator/=( const tmat3<T> & m )
			{
				return ( *this = *this * detail::compute_inverse<T>( m ) );
			}

			inline tmat3<T> & operator++()
			{
				++this->value[0];
				++this->value[1];
				++this->value[2];
				return *this;
			}

			inline tmat3<T> & operator--()
			{
				--this->value[0];
				--this->value[1];
				--this->value[2];
				return *this;
			}

			inline tmat3<T> operator++( int )
			{
				tmat3<T> result( *this );
				++*this;
				return result;
			}

			inline tmat3<T> operator--( int )
			{
				tmat3<T> result( *this );
				--*this;
				return result;
			}
		};

		namespace detail
		{
			template <typename T>
			inline tmat3<T> compute_inverse( const tmat3<T> & m )
			{
				T one_over_det = static_cast<T>( 1 ) / (
					+ m[0][0] * ( m[1][1] * m[2][2] - m[2][1] * m[1][2] )
					- m[1][0] * ( m[0][1] * m[2][2] - m[2][1] * m[0][2] )
					+ m[2][0] * ( m[0][1] * m[1][2] - m[1][1] * m[0][2] )
				);

				tmat3<T> result( no_init );
				result[0][0] = +( m[1][1] * m[2][2] - m[2][1] * m[1][2] ) * one_over_det;
				result[1][0] = -( m[1][0] * m[2][2] - m[2][0] * m[1][2] ) * one_over_det;
				result[2][0] = +( m[1][0] * m[2][1] - m[2][0] * m[1][1] ) * one_over_det;
				result[0][1] = -( m[0][1] * m[2][2] - m[2][1] * m[0][2] ) * one_over_det;
				result[1][1] = +( m[0][0] * m[2][2] - m[2][0] * m[0][2] ) * one_over_det;
				result[2][1] = -( m[0][0] * m[2][1] - m[2][0] * m[0][1] ) * one_over_det;
				result[0][2] = +( m[0][1] * m[1][2] - m[1][1] * m[0][2] ) * one_over_det;
				result[1][2] = -( m[0][0] * m[1][2] - m[1][0] * m[0][2] ) * one_over_det;
				result[2][2] = +( m[0][0] * m[1][1] - m[1][0] * m[0][1] ) * one_over_det;
			
				return result;
			}

		}

		template <typename T>
		inline tmat3<T> const operator-( const tmat3<T> & m )
		{
			return tmat3<T>(
				-m[0],
				-m[1],
				-m[2] );
		}

		template <typename T>
		inline tmat3<T> operator+( const tmat3<T> & m, T s )
		{
			return tmat3<T>(
				m[0] + s,
				m[1] + s,
				m[2] + s );
		}

		template <typename T>
		inline tmat3<T> operator+( T s, const tmat3<T> & m )
		{
			return tmat3<T>(
				m[0] + s,
				m[1] + s,
				m[2] + s );
		}

		template <typename T>
		inline tmat3<T> operator+( const tmat3<T> & m1, const tmat3<T> & m2 )
		{
			return tmat3<T>(
				m1[0] + m2[0],
				m1[1] + m2[1],
				m1[2] + m2[2] );
		}

		template <typename T>
		inline tmat3<T> operator-( const tmat3<T> & m, T s )
		{
			return tmat3<T>(
				m[0] - s,
				m[1] - s,
				m[2] - s );
		}

		template <typename T>
		inline tmat3<T> operator-( T s, const tmat3<T> & m )
		{
			return tmat3<T>(
				s - m[0],
				s - m[1],
				s - m[2] );
		}

		template <typename T>
		inline tmat3<T> operator-( const tmat3<T> & m1, const tmat3<T> & m2 )
		{
			return tmat3<T>(
				m1[0] - m2[0],
				m1[1] - m2[1],
				m1[2] - m2[2] );
		}

		template <typename T>
		inline tmat3<T> operator*( const tmat3<T> & m, T s )
		{
			return tmat3<T>(
				m[0] * s,
				m[1] * s,
				m[2] * s );
		}

		template <typename T>
		inline tmat3<T> operator*( T s, const tmat3<T> & m )
		{
			return tmat3<T>(
				m[0] * s,
				m[1] * s,
				m[2] * s );
		}

		template <typename T>
		inline typename tmat3<T>::column_type operator*( const tmat3<T> & m, const typename tmat3<T>::row_type & v )
		{
			return typename tmat3<T>::column_type(
				m[0][0] * v.x + m[1][0] * v.y + m[2][0] * v.z,
				m[0][1] * v.x + m[1][1] * v.y + m[2][1] * v.z,
				m[0][2] * v.x + m[1][2] * v.y + m[2][2] * v.z );
		}

		template <typename T>
		inline typename tmat3<T>::row_type operator*( const typename tmat3<T>::column_type & v, const tmat3<T> & m )
		{
			return typename tmat3<T>::row_type(
				m[0][0] * v.x + m[0][1] * v.y + m[0][2] * v.z,
				m[1][0] * v.x + m[1][1] * v.y + m[1][2] * v.z,
				m[2][0] * v.x + m[2][1] * v.y + m[2][2] * v.z );
		}

		template <typename T>
		inline tmat3<T> operator*( const tmat3<T> & m1, const tmat3<T> & m2 )
		{
			tmat3<T> result( no_init );
			result[0][0] = m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2];
			result[0][1] = m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2];
			result[0][2] = m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1] + m1[2][2] * m2[0][2];
			result[1][0] = m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2];
			result[1][1] = m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2];
			result[1][2] = m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1] + m1[2][2] * m2[1][2];
			result[2][0] = m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2];
			result[2][1] = m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2];
			result[2][2] = m1[0][2] * m2[2][0] + m1[1][2] * m2[2][1] + m1[2][2] * m2[2][2];
			return result;
		}

		template <typename T>
		inline tmat3<T> operator/( const tmat3<T> & m, T s )
		{
			return tmat3<T>(
				m[0] / s,
				m[1] / s,
				m[2] / s );
		}

		template <typename T>
		inline tmat3<T> operator/( T s, const tmat3<T> & m )
		{
			return tmat3<T>(
				s / m[0],
				s / m[1],
				s / m[2] );
		}

		template <typename T>
		inline typename tmat3<T>::column_type operator/( const tmat3<T> & m, const typename tmat3<T>::row_type & v )
		{
			return detail::compute_inverse<T>( m ) * v;
		}

		template <typename T>
		inline typename tmat3<T>::row_type operator/( const typename tmat3<T>::column_type & v, const tmat3<T> & m )
		{
			return v * detail::compute_inverse<T>( m );
		}

		template <typename T>
		inline tmat3<T> operator/( const tmat3<T> & m1, const tmat3<T> & m2 )
		{
			tmat3<T> m1_copy( m1 );
			return m1_copy /= m2;
		}

		template <typename T>
		inline bool operator==( const tmat3<T> & m1, const tmat3<T> & m2 )
		{
			return ( m1[0] == m2[0] ) && ( m1[1] == m2[1] ) && ( m1[2] == m2[2] );
		}

		template <typename T>
		inline bool operator!=( const tmat3<T> & m1, const tmat3<T> & m2 )
		{
			return ( m1[0] != m2[0] ) || ( m1[1] != m2[1] ) || ( m1[2] != m2[2] );
		}

	}

}

#endif // NV_STL_MATH_MAT3_HH
