// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

// TODO : Force no resize on the buffer_slice!
// TODO : Multiple update support?

#ifndef NV_GFX_SLICED_BUFFER_HH
#define NV_GFX_SLICED_BUFFER_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/vector.hh> 

namespace nv
{

	template< typename T > class sliced_buffer;

	template< typename T >
	class buffer_slice
	{
	public:
		typedef sliced_buffer<T> cache_type;
		typedef vector<T>        vector_type;
		typedef size_t           size_type;

		buffer_slice( cache_type* c ) 
			: m_cache( c ), m_offset( 0 ), m_cached_size( 0 ), m_locked( true )
		{
			m_cache->reset();
		}
		bool commit( bool force_changed = false )
		{
			bool resized = force_changed || ( m_cached_size != m_data.size() );
			bool result  = m_cache->commit( m_data, m_locked, resized, m_offset );
			m_cached_size = m_data.size();
			m_locked      = false;
			return result;
		}
		
		template < typename V >
		bool commit_offset( bool force_changed, V value )
		{
			bool locked = m_locked;
			bool result = commit( force_changed );
			if ( locked || result )
			{
				// funny, but this is what GCC expects
				m_cache->template add_offset<T>( m_data.size(), m_offset, static_cast<T>(value) );
			}
			return result;
		}

		bool is_locked() const
		{
			return m_locked;
		}

		size_t get_offset() const
		{
			return m_offset;
		}

		vector_type& lock()
		{
			m_locked = true;
			return m_data;
		}
		const vector_type& data() 
		{ 
			return m_data; 
		}
		~buffer_slice()
		{
			m_cache->reset();
		}
	public:
		vector_type m_data;
		cache_type* m_cache;
		size_type   m_offset;
		size_type   m_cached_size;
		bool        m_locked;
	};

	template< typename T >
	class sliced_buffer
	{
	public:
		typedef vector<T>  vector_type;
		typedef T          value_type;
		static const size_t value_type_size = sizeof(T);

		sliced_buffer( context* ctx, buffer_type type, buffer_hint hint, size_t initial_size ) 
			: m_context( ctx )
			, m_buffer()
			, m_type( type )
			, m_hint( hint )
			, m_full_update( true )
			, m_data()
			, m_capacity(0)
			, m_min(0)
			, m_max(0)
		{ 
			create_buffer( initial_size );
		}

		bool commit( const vector_type& bv, bool updated, bool resized, size_t& offset )
		{
			if ( !m_full_update && resized )
			{
				m_data.erase( m_data.begin() + static_cast<int>( offset ), m_data.end() );
				m_min = nv::min<size_t>( m_min, offset );
				m_full_update = true;
			}
			if ( m_full_update )
			{
				offset = m_data.size();
				m_data.append( bv.cbegin(), bv.cend() );
			}
			else if ( updated )
			{
//				raw_copy( bv.cbegin(), bv.cend(), m_data.begin() + (int)offset );
				raw_copy( bv.data(), bv.data() + bv.size(), m_data.data() + static_cast<int>( offset ) );
				m_min = nv::min<size_t>( m_min, offset );
				m_max = nv::max<size_t>( m_max, offset + bv.size() );
			}
			return m_full_update;
		}

		template < typename V > 
		void add_offset( size_t size, size_t offset, V value )
		{
			if ( size == 0 ) return;
			T* ptr  = m_data.data() + offset;
			T* pend = ptr + size;
			for ( ; ptr != pend; ptr++ )
			{
				*ptr += value;
			}
		}

		void reset()
		{
			m_data.clear();
			m_full_update = true;
			m_min = 0;
			m_max = 0;
		}

		/**
		 * Returns true if buffer has been recreated
		 */
		bool commit()
		{
			bool result = false;
			size_t bsize = get_max_size();

			if ( m_data.size() > bsize )
			{
				while( m_data.size() > bsize ) bsize *= 2;
				create_buffer( bsize );
				m_full_update = true;
				m_min  = 0;
				result = true;
			}
			if ( m_full_update ) m_max = m_data.size();
			if ( m_max > 0 )
			{
				size_t offset = m_min * value_type_size;
				size_t size   = (m_max-m_min) * value_type_size;
				m_context->update( m_buffer, m_data.data() + m_min, offset, size );
			}
			m_full_update = false;
			m_min = get_max_size();
			m_max = 0;
			return result;
		}

		size_t get_max_size() const
		{ 
			return m_capacity;
		}

		size_t get_size() const
		{ 
			return m_data.size(); 
		}

		buffer get_buffer()
		{ 
			return m_buffer; 
		}

		virtual ~sliced_buffer() 
		{
			m_context->release( m_buffer );
		}
	private:
		void create_buffer( size_t size )
		{
			m_capacity = size;
			if ( m_buffer )
				m_context->create_buffer( m_buffer, size * value_type_size, nullptr );
			else
				m_buffer = m_context->create_buffer( m_type, m_hint, size * value_type_size, nullptr );
		}
	private:
		context*    m_context;
		buffer      m_buffer;
		buffer_type m_type;
		buffer_hint m_hint;
		bool        m_full_update;
		vector_type m_data;
		uint32      m_capacity;

		size_t      m_min;
		size_t      m_max;
	};

	template< typename T, typename I = uint16 > class indexed_sliced_buffer;

	template< typename T, typename I = uint16 >
	class indexed_buffer_slice
	{
	public:
		typedef indexed_sliced_buffer<T,I> cache;
		typedef buffer_slice<T> vertex_slice;
		typedef buffer_slice<I> index_slice;
		typedef typename vertex_slice::vector_type vertex_vector;
		typedef typename index_slice::vector_type  index_vector;

		indexed_buffer_slice( cache* c )
			: m_vertex_slice( &c->get_vertex_cache() )
			, m_index_slice( &c->get_index_cache() )
			, m_cache( c )
		{
		}

		void commit()
		{
			bool changed = m_vertex_slice.commit();
			m_index_slice.commit_offset( changed, m_vertex_slice.get_offset() );
		}
		vertex_vector& lock_vertices()
		{
			return m_vertex_slice.lock();
		}
		index_vector& lock_indices()
		{
			return m_index_slice.lock();
		}
		const vertex_vector& vertex_data() 
		{ 
			return m_vertex_slice.data();
		}
		const index_vector& index_data() 
		{ 
			return m_index_slice.data();
		}

	private:
		vertex_slice m_vertex_slice;
		index_slice  m_index_slice;
		cache*       m_cache;
	};

	template< typename T, typename I >
	class indexed_sliced_buffer
	{
	public:
		typedef sliced_buffer<T> sliced_vertex_buffer;
		typedef sliced_buffer<I> sliced_index_buffer;
		static const int value_type_size = sizeof(T);

		indexed_sliced_buffer( context* ctx, buffer_hint hint, int initial_size, int initial_index_size )
			: m_vertex_buffer( ctx, VERTEX_BUFFER, hint, initial_size )
			, m_index_buffer( ctx, INDEX_BUFFER, hint, initial_index_size )
		{

		}

		void reset()
		{
			m_vertex_buffer.reset();
			m_index_buffer.reset();
		}

		bool commit()
		{
			bool vresult = m_vertex_buffer.commit();
			bool iresult = m_index_buffer.commit();
			return vresult || iresult;
		}

		int get_max_vertex_size() const { return m_vertex_buffer.get_max_size(); }
		int get_max_index_size()  const { return m_index_buffer.get_max_size(); }
		int get_vertex_size()     const { return m_vertex_buffer.get_size(); }
		int get_index_size()      const { return m_index_buffer.get_size(); }
		buffer get_vertex_buffer() { return m_vertex_buffer.get_buffer(); }
		buffer get_index_buffer()  { return m_index_buffer.get_buffer(); }
		sliced_vertex_buffer& get_vertex_cache() { return m_vertex_buffer; }
		sliced_index_buffer&  get_index_cache()  { return m_index_buffer; }

	private:
		sliced_vertex_buffer m_vertex_buffer;
		sliced_index_buffer  m_index_buffer;
	};


} // namespace nv

#endif // NV_GFX_SLICED_BUFFER_HH
