// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file priority_queue.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief priority queue adaptor
*/

#ifndef NV_STL_PRIORITY_QUEUE_HH
#define NV_STL_PRIORITY_QUEUE_HH

#include <nv/stl/vector.hh>

namespace nv
{

	template < 
		typename T,
		typename Container = vector< T >,
		typename Predicate = less< typename Container::value_type >
	>
	class priority_queue
	{
	public:
		typedef priority_queue< T, Container, Predicate > this_type;
		typedef Predicate                                 compare_type;
		typedef Container                                 container_type;
		typedef typename Container::value_type            value_type;
		typedef typename Container::size_type             size_type;
		typedef typename Container::reference             reference;
		typedef typename Container::const_reference       const_reference;

		priority_queue() {}
		priority_queue( const this_type& ) = delete;
		priority_queue( this_type&& other ) 
			: m_data( ::nv::move( other.m_data ) )
			, m_compare( ::nv::move( other.m_compare ) ) {}

		this_type& operator=( const this_type& ) = delete;
		this_type& operator=( this_type&& )
		{
			m_data    = ::nv::move( m_data );
			m_compare = ::nv::move( m_compare );
			return ( *this );
		}

		explicit priority_queue( const compare_type& pred ) : m_data(), m_compare( pred ) {}
		explicit priority_queue( container_type&& container, const compare_type& pred = compare_type() )
			: m_data( ::nv::move( container ) )
			, m_compare( pred )
		{
			make_heap( m_data.begin(), m_data.end() );
		}

		template < typename InputIterator >
		priority_queue( InputIterator first, InputIterator last )
			: m_data( first, last )
		{
			make_heap( m_data.begin(), m_data.end() );
		}

		template < typename InputIterator >
		priority_queue( InputIterator first, InputIterator last, const compare_type& pred )
			: m_data( first, last ), m_compare( pred )
		{
			make_heap( m_data.begin(), m_data.end() );
		}

		void push( value_type&& value )
		{
			m_data.push_back( ::nv::move( value ) );
			push_heap( m_data.begin(), m_data.end(), m_compare );
		}

		template< typename... Args >
		void emplace( Args&&... args )
		{
			m_data.emplace_back( ::nv::forward<Args>( args )... );
			push_heap( m_data.begin(), m_data.end(), m_compare );
		}

		bool empty() const
		{
			return m_data.empty();
		}

		size_type size() const
		{
			return m_data.size();
		}

		const_reference top() const
		{
			return m_data.front();
		}

		void push( const value_type& value )
		{
			m_data.push_back( value );
			push_heap( m_data.begin(), m_data.end(), m_compare );
		}

		void pop()
		{
			pop_heap( m_data.begin(), m_data.end(), m_compare );
			m_data.pop_back();
		}

		void swap( this_type& rhs )
		{
			::nv::swap( m_data, rhs.m_data );
			::nv::swap( m_compare, rhs.m_compare );
		}

	protected:
		container_type m_data;
		compare_type   m_compare;
	};

	template <
		typename T,
		typename Container,
		typename Predicate
	>
	inline void swap(
		priority_queue< T, Container, Predicate >& lhs,
		priority_queue< T, Container, Predicate >& rhs
	)
	{
		lhs.swap( rhs );
	}

}

#endif // NV_STL_PRIORITY_QUEUE_HH
