// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file animation.hh
 * @author Kornel Kisielewicz
 * @brief animation
 */

#ifndef NV_ENGINE_LIGHT_HH
#define NV_ENGINE_LIGHT_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/stl/math.hh>
#include <nv/stl/array.hh>
#include <nv/interface/context.hh>

namespace nv
{

	enum class shadow_type
	{
		NONE,
		ACTIVE,
		PASSIVE,
	};

	struct light_data
	{
		vec3 position;
		vec4 color;
		vec4 direction;
		float angle;
		float umbra_angle;
		float range;
		uint32 smooth;
		shadow_type shadow;

		light_data() : color( 1.0 ), direction( 0.0f, -1.0f, 0.0f, 1.0f ), angle( 0.6f ), umbra_angle( 0.2f ), range( 4.0 ), smooth( 0 ), shadow( shadow_type::ACTIVE ) {}
		light_data( const vec3& a_position, const vec3& a_color, shadow_type type )
			: position( a_position ), color( a_color, 1.0f )
			, direction( 0.0f, -1.0f, 0.0f, 1.0f ), angle( 0.6f ), umbra_angle( 0.2f ), range( 4.0 ), smooth( 0 ), shadow( type ) {}
		light_data( const light_data& ) = default;
	};
	
	NV_RTTI_DECLARE( nv::light_data )

	struct light_data_distance_compare
	{
		vec3 source;
		light_data_distance_compare( const vec3& source ) : source( source ) {}
		bool operator()( const light_data* l, const light_data* r )
		{
			vec3 dl( l->position - source );
			vec3 dr( r->position - source );
			return math::length_sq( dl ) < math::length_sq( dr );
		}
	};

	class gpu_light_block
	{
	public:
		gpu_light_block() : m_buffer_index(0), m_max_size(0) {}
		const camera& get_camera( uint32 i )       const { return m_cameras[i]; }
		vec3          get_position( uint32 i )     const { return vec3( m_blocks[i].position ); }
		const vec4&   get_shadow_index( uint32 i ) const { return m_blocks[i].shadow; }
		uint32 index() const { return m_buffer_index; }

		void initialize( context* ctx, uint32 max_size, uint32 index )
		{
			m_buffer_index = index;
			m_max_size = max_size;
			m_cameras.resize( m_max_size );
			m_blocks.resize( m_max_size );
			if ( m_buffer ) ctx->release( m_buffer );
			m_buffer = ctx->create_buffer( UNIFORM_BUFFER, STREAM_DRAW, sizeof( block_data ) * m_max_size );
		}

		void update( array_view< light_data* > lights )
		{
			uint32 count = nv::min< uint32 >( lights.size(), m_max_size );
			uint32 shadow_counter[4] = { 0, 0, 0, 0 };
			for ( size_t i = 0; i < count; ++i )
			{
				m_cameras[i].set_lookat( vec3(), vec3( lights[i]->direction ), vec3( 1.0f, 0.0f, 0.0f ) );
				m_cameras[i].set_perspective( 150.f, 1.0f, 0.01f, 5.0f );

				m_blocks[i].position  = vec4( lights[i]->position, 1.0f );
				m_blocks[i].color     = vec4( vec3( lights[i]->color ) * lights[i]->color.w, 1.0f );
				m_blocks[i].direction = lights[i]->direction;
				m_blocks[i].params    = vec4( lights[i]->angle, lights[i]->umbra_angle, lights[i]->range, 1.0f );
				m_blocks[i].mvp       = m_cameras[i].get_projection() * m_cameras[i].get_view();
				uint32 index = lights[i]->smooth;
				m_blocks[i].shadow    = vec4( float( index ), float( shadow_counter[index]++ ), 0.0f, 0.0f );
			}
		}

		void bind( context* ctx ) const
		{
			ctx->bind( m_buffer, m_buffer_index );
		}

		void update( context* ctx )
		{
			ctx->update( m_buffer, m_blocks.data(), 0, sizeof( block_data ) * m_max_size );
			ctx->bind( m_buffer, m_buffer_index );
		}

	private:
		struct block_data
		{
			vec4 position;
			vec4 color;
			vec4 direction;
			vec4 params; // 0 - angle, 1 - umbra angle, 2 - attenuation
			mat4 mvp;
			vec4 shadow;
		};

		dynamic_array< camera >     m_cameras;
		dynamic_array< block_data > m_blocks;
		buffer m_buffer;
		uint32 m_buffer_index;
		uint32 m_max_size;
	};

	struct gpu_light_geometry
	{
	public:
		static const unsigned lgeo_vertices = 4;
		static const unsigned lgeo_indices = 6;

		void initialize( uint32 max_size )
		{
			m_max_size = max_size;
			m_light_data.resize( lgeo_vertices * m_max_size );
			m_light_idata.resize( lgeo_indices * m_max_size );
		}
		uint32 max_size() const { return m_max_size; }

		vertex_array get_va() const { return m_va; }
		uint32 index_stride() const { return lgeo_indices; }

		void update( array_view< light_data* > lights )
		{
			uint32 count = nv::min< uint32 >( lights.size(), m_max_size );
			for ( size_t i = 0; i < count; ++i )
			{
				vec3 nlp = lights[i]->position;
				float radius = lights[i]->range;
				nlp.y = 0.1f;
				vec3 r1( radius, 0.0f, radius );
				vec3 r2( radius, 0.0f, -radius );

				m_light_data[i * lgeo_vertices + 0] = light_vertex{ vec4( nlp - r1, float( i ) ) };
				m_light_data[i * lgeo_vertices + 1] = light_vertex{ vec4( nlp - r2, float( i ) ) };
				m_light_data[i * lgeo_vertices + 2] = light_vertex{ vec4( nlp + r1, float( i ) ) };
				m_light_data[i * lgeo_vertices + 3] = light_vertex{ vec4( nlp + r2, float( i ) ) };

				uint16 indices[lgeo_indices] = { 0, 1, 2, 2, 3, 0 };
				for ( uint16 j = 0; j < lgeo_indices; ++j )
				{
					m_light_idata[i * lgeo_indices + j] = uint16( indices[j] + i * lgeo_vertices );
				}
			}
		}


		void update( context* ctx, uint32 count )
		{
			if ( m_va.is_nil() )
			{
				// remove data()
				m_position_buffer = ctx->create_buffer( VERTEX_BUFFER, STREAM_DRAW, m_light_data.size() * sizeof( light_vertex ), m_light_data.data() );
				m_index_buffer = ctx->create_buffer( INDEX_BUFFER, STREAM_DRAW, m_light_idata.size() * sizeof( light_index ), m_light_idata.data() );
				vertex_array_desc va_desc;
				va_desc.add_vertex_buffers< light_vertex >( m_position_buffer, true );
				va_desc.set_index_buffer( m_index_buffer, type_to_enum< light_index >::type, true );
				m_va = ctx->create_vertex_array( va_desc );
			}
			else
			{
				ctx->update( m_position_buffer, m_light_data.data(), 0, lgeo_vertices * count * sizeof( light_vertex ) );
				ctx->update( m_index_buffer, m_light_idata.data(), 0, lgeo_indices  * count * sizeof( uint16 ) );
			}

		}

	private:
		struct light_vertex
		{
			vec4  position;  // position, w = index
		};
		typedef uint16 light_index;

		uint32       m_max_size;
		buffer       m_position_buffer;
		buffer       m_index_buffer;
		vertex_array m_va;

		dynamic_array< light_vertex > m_light_data;
		dynamic_array< light_index >  m_light_idata;

	};

	class gpu_light_data
	{
	public:
		void initialize( context* ctx, uint32 max_size, uint32 index )
		{
			m_active = 0;
			m_block.initialize( ctx, max_size, index );
			m_geometry.initialize( max_size );
		}

		void update( context* ctx, array_view< light_data* > lights )
		{
			m_active = nv::min<uint32>( m_geometry.max_size(), lights.size() );
			if ( m_active == 0 ) return;
			m_block.update( lights );
			m_geometry.update( lights );

			m_block.update( ctx );
			m_geometry.update( ctx, m_active );
		}

		void filter_update( context* ctx, array_view< light_data* > lights, shadow_type type )
		{
			uint32 count = 0;
			array< light_data*, 128 > filtered_lights;
			uint32 size = nv::min< uint32 >( lights.size(), geometry().max_size() );

			for ( auto light : lights )
			{
				if ( light->shadow == type )
					filtered_lights[count++] = light;
				if ( count >= size ) break;
			}
			update( ctx, array_view< light_data* >( filtered_lights.data(), count ) );
		}

		const gpu_light_block& block() const { return m_block; }
		const gpu_light_geometry& geometry() const { return m_geometry; }
		uint32 active_count() const { return m_active; }

	protected:
		uint32             m_active;
		gpu_light_block    m_block;
		gpu_light_geometry m_geometry;
	};


}

#endif // NV_ENGINE_LIGHT_HH

