// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_LUA_VALUES_HH
#define NV_LUA_VALUES_HH

#include <nv/common.hh>
#include <nv/stl/type_traits/properties.hh>
#include <nv/stl/string.hh>

struct lua_State;

namespace nv
{
	namespace lua
	{
		class ref
		{
		public:
			static const int none  = -2;
			static const int nil   = -1;

			ref() : m_value( nil ) {}
			explicit ref( int lua_ref ) : m_value( lua_ref ) {}
			bool is_valid() const { return m_value >= 0; }
			int get() const { return m_value; }
		protected:
			int m_value;
		};

		typedef ptrdiff_t     linteger;
		typedef unsigned long lunsigned;
		typedef double        lnumber;
		typedef int( __cdecl *lfunction ) ( struct lua_State *L );


		template < typename T, class Enable = void >
		struct pass_traits;

		namespace detail
		{
			int upvalue_index( int i );

			void push_nil        ( lua_State *L );
			void push_unsigned   ( lua_State *L, lunsigned v );
			void push_integer    ( lua_State *L, linteger v );
			void push_number     ( lua_State *L, lnumber v );
			void push_bool       ( lua_State *L, bool v );
			void push_string_view( lua_State *L, string_view s );
			void push_pointer    ( lua_State *L, void* p );
			void push_ref_object ( lua_State *L, ref object );

			lunsigned   to_unsigned   ( lua_State *L, int index );
			linteger    to_integer    ( lua_State *L, int index );
			lnumber     to_number     ( lua_State *L, int index );
			bool        to_bool       ( lua_State *L, int index );
			string_view to_string_view( lua_State *L, int index );
			void*       to_pointer    ( lua_State *L, int index );
			void*       to_ref_object ( lua_State *L, int index );

			lunsigned   to_unsigned   ( lua_State *L, int index, lunsigned def );
			linteger    to_integer    ( lua_State *L, int index, linteger def );
			lnumber     to_number     ( lua_State *L, int index, lnumber def );
			bool        to_bool       ( lua_State *L, int index, bool def );
			string_view to_string_view( lua_State *L, int index, string_view def );
			void*       to_pointer    ( lua_State *L, int index, void* def );
			void*       to_ref_object ( lua_State *L, int index, void* def );

			void pop_and_discard( lua_State *L, int count );
			bool is_userdata( lua_State *L, int index, const char* metatable );
			void* raw_check_userdata( lua_State *L, int index, const char* metatable );
			void* raw_allocate_userdata( lua_State *L, size_t size, const char* metatable );

			template <typename T>
			void push_userdata( lua_State *L, const T& v, const char* metatable )
			{
				new ( raw_allocate_userdata( L, sizeof( T ), metatable ) ) T( v );
			}

			template <typename T>
			T* to_userdata( lua_State *L, int index, const char* metatable )
			{
				return reinterpret_cast<T*>( raw_check_userdata( L, index, metatable ) );
			}

			template <typename T>
			T* to_userdata( lua_State *L, int index )
			{
				return reinterpret_cast<T*>( raw_check_userdata( L, index, pass_traits<T>::metatable() ) );
			}
		}

		template < typename T >
		struct metatable_pass_traits
		{
			typedef T value_type;
			static void push( lua_State *L, const value_type& v )
			{
				detail::push_userdata< value_type >( L, v, pass_traits< value_type >::metatable() );
			}
			static value_type to( lua_State *L, int index )
			{
				// TODO: ASSERT?
				value_type* result = detail::to_userdata< value_type >( L, index, pass_traits< value_type >::metatable() );
				return result ? *result : value_type();
			}
			static value_type to( lua_State *L, int index, const value_type& def )
			{
				value_type* result = detail::to_userdata< value_type >( L, index, pass_traits< value_type >::metatable() );
				return result ? *result : def;
			}
		};

		template <>
		struct pass_traits<linteger>
		{
			static void push( lua_State *L, linteger s ) { detail::push_integer( L, s ); }
			static linteger to( lua_State *L, int index ) { return detail::to_integer( L, index ); }
			static linteger to( lua_State *L, int index, linteger def ) { return detail::to_integer( L, index, def ); }
		};

		template <>
		struct pass_traits<lunsigned>
		{
			static void push( lua_State *L, lunsigned s ) { detail::push_unsigned( L, s ); }
			static lunsigned to( lua_State *L, int index ) { return detail::to_unsigned( L, index ); }
			static lunsigned to( lua_State *L, int index, lunsigned def ) { return detail::to_unsigned( L, index, def ); }
		};

		template <>
		struct pass_traits<lnumber>
		{
			static void push( lua_State *L, lnumber s ) { detail::push_number( L, s ); }
			static lnumber to( lua_State *L, int index ) { return detail::to_number( L, index ); }
			static lnumber to( lua_State *L, int index, lnumber def ) { return detail::to_number( L, index, def ); }
		};

		template <>
		struct pass_traits<float>
		{
			static void push( lua_State *L, float s ) { detail::push_number( L, static_cast<double>( s ) ); }
			static float to( lua_State *L, int index ) { return static_cast<float>( detail::to_number( L, index ) ); }
			static float to( lua_State *L, int index, lnumber def ) { return static_cast<float>( detail::to_number( L, index, def ) ); }
		};

		template <>
		struct pass_traits<bool>
		{
			static void push( lua_State *L, bool s ) { detail::push_bool( L, s ); }
			static bool to( lua_State *L, int index ) { return detail::to_bool( L, index ); }
			static bool to( lua_State *L, int index, bool def ) { return detail::to_bool( L, index, def ); }
		};

		template <>
		struct pass_traits < string_view >
		{
			static void push( lua_State *L, string_view s ) { detail::push_string_view( L, s ); }
			static string_view to( lua_State *L, int index ) { return detail::to_string_view( L, index ); }
			static string_view to( lua_State *L, int index, string_view def ) { return detail::to_string_view( L, index, def ); }
		};

		template <>
		struct pass_traits < const_string >
		{
			static void push( lua_State *L, const const_string& s ) { detail::push_string_view( L, s ); }
			static const_string to( lua_State *L, int index ) { return const_string( detail::to_string_view( L, index ) ); }
			static const_string to( lua_State *L, int index, const string_view& def ) { return const_string( detail::to_string_view( L, index, def ) ); }
		};

		// TODO: might be better to pass only string_views? 
		//       or maybe make them all decay to them?
		template < size_t N >
		struct pass_traits < short_string<N> >
		{
			typedef short_string<N> value_type;
			static void push( lua_State *L, const value_type& s ) { detail::push_string_view( L, s ); }
			static value_type to( lua_State *L, int index ) { return value_type( detail::to_string_view( L, index ) ); }
			static value_type to( lua_State *L, int index, const string_view& def ) { return value_type( detail::to_string_view( L, index, def ) ); }
		};

		template <>
		struct pass_traits < string_buffer >
		{
			typedef string_buffer value_type;
			static void push( lua_State *L, const value_type& s ) { detail::push_string_view( L, s ); }
			static value_type to( lua_State *L, int index ) { return value_type( detail::to_string_view( L, index ) ); }
			static value_type to( lua_State *L, int index, const string_view& def ) { return value_type( detail::to_string_view( L, index, def ) ); }
		};

		template <>
		struct pass_traits<ref>
		{
			static void push( lua_State *L, ref s ) { detail::push_ref_object( L, s ); }
		};


		namespace detail
		{

			template <typename T, class Enable = void >
			struct lua_type_impl
			{
				typedef T type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_floating_point< T >::value > >
			{
				typedef T type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_integral< T >::value && is_signed< T >::value > >
			{
				typedef linteger type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_integral< T >::value && is_unsigned< T >::value > >
			{
				typedef lunsigned type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_cstring< T >::value > >
			{
				typedef string_view type;
			};

			template <>
			struct lua_type_impl< bool >
			{
				typedef bool type;
			};

		}

		template < typename T >
		struct converted_type
		{
			typedef typename detail::lua_type_impl< remove_cvr_t<T> >::type type;
		};

		template < typename T >
		using converted_type_t = typename converted_type< T >::type;

		namespace detail
		{

 			template < typename T >
 			void push_value( lua_State *L, T&& p )
 			{
				pass_traits< converted_type_t<T> >::push( L, ::nv::forward<T>( p ) );
 			}

			template < typename T >
			void push_values(lua_State *L, T&& p)
			{
				push_value( L, ::nv::forward<T>(p) );
			}
			template < typename T, typename ...Ts >
			void push_values(lua_State *L, T&& p, Ts&& ...ps)
			{
				push_value(L, ::nv::forward<T>(p));
				push_values(L, ::nv::forward<Ts>(ps)...);
			}


			template < typename T >
			inline void pop_value( lua_State *L, T& p )
			{
				p = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1 ) );
				detail::pop_and_discard(L, 1);
			}
			template < typename T >
			inline T pop_return_value( lua_State *L )
			{
				T ret = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1 ) );
				detail::pop_and_discard(L, 1);
				return ret;
			}

			template <>
			inline void pop_return_value<void>( lua_State* )
			{
			}

			template < typename T >
			inline void pop_value( lua_State *L, T& p, const T& def )
			{
				p = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1, def ) );
				detail::pop_and_discard(L, 1);
			}
			template < typename T >
			inline T pop_return_value( lua_State *L, const T& def )
			{
				T ret = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1, def ) );
				detail::pop_and_discard(L, 1);
				return ret;
			}

			template < typename T >
			inline converted_type_t<T> get_value( lua_State *L, int index )
			{
				return pass_traits< converted_type_t<T> >::to( L, index );
			}

			template < typename T >
			inline converted_type_t<T> get_value( lua_State *L, int index, const T& def )
			{
				return pass_traits< converted_type_t<T> >::to( L, index, def );
			}

			template < typename T >
			struct return_count
			{
				const static int value = 1;
			};

			template <>
			struct return_count<void>
			{
				const static int value = 0;
			};

		}

	} // namespace lua

} // namespace nv

#endif // NV_LUA_VALUES_HH
