// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file image_manager.hh
* @author Kornel Kisielewicz
* @brief image_manager
*/

#ifndef NV_ENGINE_RENDERER_HH
#define NV_ENGINE_RENDERER_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/interface/device.hh>
#include <nv/interface/context.hh>
#include <nv/engine/program_manager.hh>
#include <nv/engine/material_manager.hh>
#include <nv/engine/mesh_manager.hh>
#include <nv/engine/render_pass.hh>

namespace nv
{

	struct renderer_element
	{
		resource< gpu_mesh >     mesh;
		resource< gpu_material > mat;
		uint32                   layer;
		sint32                   bone_offset;
		transform                model;
		flags< 32 >              flags;
	};

	struct renderer_direct_element
	{
		vertex_array             va;
		uint32                   count;
		uint32                   instances;
		uint32                   first;
		resource< gpu_material > material;
		resource< program >      programs[4];
		transform                model;
		flags< 32 >              flags;
	};

	struct renderer_element_compare
	{
		bool operator()( const renderer_element& l, const renderer_element& r )
		{
			if ( l.layer < r.layer ) return true;
			if ( l.layer > r.layer ) return false;
			if ( *l.flags.data() < *r.flags.data() ) return true;
			if ( *l.flags.data() > *r.flags.data() ) return false;
			if ( l.mat < r.mat ) return true;
			if ( l.mat > r.mat ) return false;
			return false;
		}
	};

	class renderer
	{
	public:
		enum stats
		{
			DRAW_CALL,
			TRIANGLE_COUNT,
			MAX
		};

		explicit renderer( context* ctx );
		void start_frame()
		{
			for ( uint32& s : m_statistics ) s = 0;
		}

		void push_element( const nv::renderer_element& element )
		{
			m_elements.push_back( element );
		}

		void sort();
		void render( const scene_state& ss, const render_pass& pass );
		void render( const scene_state& ss, const render_pass& pass, vertex_array va, uint32 va_count, uint32 program_idx = 0 );
		void render_quad( const scene_state& ss, const render_pass& pass, uint32 program_idx = 0 )
		{
			render( ss, pass, m_quad, 6, program_idx );
		}
		void render_direct( const scene_state& ss, const render_pass& pass, const nv::array_view< renderer_direct_element >& elements, uint32 program_idx = 0 );

		uint32 get_stat( stats s ) const { return m_statistics[s]; }

		void end_frame()
		{
			m_elements.clear();
		}
		void reset()
		{
			m_elements.clear();
		}

	protected:
		bool match( flags<32> f, const render_pass& pass )
		{
			if ( ( f & pass.any ).empty() ) return false;
			if ( ( f & pass.exclude ).any() ) return false;
			return true;
		}

		context*                    m_context;
		vertex_array                m_quad;
		vector< renderer_element >  m_elements;
		uint32                      m_statistics[MAX];

	};

}

#endif // NV_ENGINE_RENDERER_HH