#include <nv/core/common.hh>
#include <iomanip>
#include <nv/gfx/keyframed_mesh.hh>
#include <nv/gl/gl_device.hh>
#include <nv/gfx/image.hh>
#include <nv/interface/context.hh>
#include <nv/interface/window.hh>
#include <nv/interface/mesh_loader.hh>
#include <nv/io/c_file_system.hh>
#include <nv/formats/md3_loader.hh>
#include <nv/core/profiler.hh>
#include <nv/core/logging.hh>
#include <nv/core/logger.hh>
#include <nv/core/math.hh>
#include <nv/core/time.hh>
#include <nv/core/string.hh>
#include <glm/gtx/rotate_vector.hpp>
#include <glm/gtc/matrix_access.hpp>
#include <glm/gtx/matrix_interpolation.hpp>

bool GPU_ANIMATION = true;

class mesh_part
{
public:
	mesh_part( const std::string& path, nv::window* window )
	{
		NV_PROFILE("mesh_construct");
		nv::md3_loader* loader;
		{
			NV_PROFILE("loader->load");
			nv::c_file_system fs;
			nv::stream* mesh_file = fs.open( path.c_str() );
			loader = new nv::md3_loader();
			loader->load( *mesh_file );
			delete mesh_file;
		}

		{
			NV_PROFILE("create_mesh_data");
			m_mesh_data = loader->release_mesh_data_pack();
		}
		delete loader;
		m_entry = nullptr;
		{
			NV_PROFILE("create_mesh");
			if ( GPU_ANIMATION )
				m_mesh      = new nv::keyframed_mesh_gpu( window->get_context(), m_mesh_data->get_mesh(0), m_mesh_data->get_nodes() );
			else
				m_mesh      = new nv::keyframed_mesh_cpu( window->get_context(), m_mesh_data->get_mesh(0), m_mesh_data->get_nodes() );
		}

	}

	nv::mat4 get_transform( nv::uint32 id )
	{
		return m_mesh->get_node_transform( id ).extract();
	}

	void setup_animation( nv::uint32 start, nv::uint32 stop, nv::uint32 fps, bool loop )
	{
		delete m_entry;
		m_entry = new nv::animation_entry("test", loop, fps, (float)start, float(stop-1) );
		m_mesh->run_animation( m_entry );
	}

	void update( nv::uint32 ms )
	{
		m_mesh->update_animation( m_entry, ms );
	}

	void update( nv::program program )
	{
		m_mesh->update( program );
	}

	nv::mesh_interface* get_mesh() const { return m_mesh; }

	size_t get_max_frames()
	{
		return m_mesh->get_max_frames();
	}

	~mesh_part()
	{
		delete m_mesh_data;
		delete m_mesh;
	}

private:
	nv::mesh_data_pack*  m_mesh_data;
	nv::keyframed_mesh*  m_mesh;
	nv::animation_entry* m_entry;
};

class application
{
public:
	application();
	bool initialize();
	bool run();
	~application();
protected:
	nv::device*       m_device;
	nv::context*      m_context;
	nv::window*       m_window;
	nv::texture       m_diffuse;
	nv::texture       m_diffuse_weapon;

	nv::clear_state   m_clear_state;
	nv::render_state  m_render_state;
	nv::scene_state   m_scene_state;

	mesh_part* m_torso;
	mesh_part* m_legs;
	mesh_part* m_head;
	mesh_part* m_weapon;
	nv::program m_program;
};

application::application()
{
	NV_PROFILE( "app_construct" );
	m_device  = new nv::gl_device();
	m_window  = m_device->create_window( 800, 600, false );
	m_context = m_window->get_context();

	nv::sampler sampler( nv::sampler::LINEAR, nv::sampler::REPEAT );
	nv::image_data* data = m_device->create_image_data( "data/doom.png" );
	m_diffuse  = m_device->create_texture( data->get_size(), nv::image_format( nv::RGBA, nv::UBYTE ), sampler, (void*)data->get_data() );
	delete data;
	data = m_device->create_image_data( "data/rocketl.png" );
	m_diffuse_weapon = m_device->create_texture( data->get_size(), nv::image_format( nv::RGBA, nv::UBYTE ), sampler, (void*)data->get_data() );
	delete data;

	m_clear_state.buffers = nv::clear_state::COLOR_AND_DEPTH_BUFFER;
	m_render_state.depth_test.enabled = true;
	m_render_state.culling.enabled    = true;
	m_render_state.blending.enabled   = false;
	m_render_state.blending.src_rgb_factor   = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_rgb_factor   = nv::blending::ONE_MINUS_SRC_ALPHA;
	m_render_state.blending.src_alpha_factor = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_alpha_factor = nv::blending::ONE_MINUS_SRC_ALPHA;
}

bool application::initialize()
{
	NV_PROFILE( "app_initialize" );
	m_program = m_device->create_program( 
		nv::slurp( GPU_ANIMATION ? "md3_gpu.vert" : "md3_cpu.vert" ), 
		nv::slurp( "md3.frag" ) 
	);
	m_torso   = new mesh_part( "data/upper.md3", m_window );
	m_legs    = new mesh_part( "data/lower.md3", m_window );
	m_head    = new mesh_part( "data/head.md3", m_window );
	m_weapon  = new mesh_part( "data/rocketl.md3", m_window );
	return true;
}

bool application::run()
{
	nv::profiler::pointer()->log_report(); 
	NV_PROFILE( "app_run" );
	int keypress = 0;

	glm::vec3 move( 0, 0, 0 );

	nv::uint32 ticks   = m_device->get_ticks();
	nv::uint32 last_ticks;
	nv::fps_counter_class< nv::system_us_timer > fps_counter;

	m_legs->setup_animation( 0, m_legs->get_max_frames(), 20, true );
	m_torso->setup_animation( 0, m_torso->get_max_frames(), 20, true );

	bool common = true;
	while(!keypress) 
	{
		last_ticks = ticks;
		ticks      = m_device->get_ticks();
		m_torso->update( ticks );
		m_legs->update( ticks );
		{
			NV_PROFILE( "clear" );
			m_context->clear( m_clear_state );
		}

		glm::mat4 view;
		glm::mat4 projection;
		{
			NV_PROFILE( "update_sh" );

			glm::vec3 source( 80.0f, 0.0f, 0.0f );
			glm::vec3 eye = glm::rotate( source, (ticks / 20.f), glm::vec3( 0.0,1.0,0.0 ) );

			m_scene_state.get_camera().set_lookat(eye + move, glm::vec3(0.0f, 0.0f, 0.0f) + move, glm::vec3(0.0, 1.0, 0.0));
			m_scene_state.get_camera().set_perspective(60.0f, 1.0f*800.0f/600.0f, 0.1f, 1000.0f);

			m_context->bind( m_diffuse, nv::TEX_DIFFUSE );
			m_device->set_uniform( m_program, "light_position", glm::vec3(120.0, 120.0, 0) );
			m_device->set_uniform( m_program, "light_diffuse",  glm::vec4(1.0,1.0,1.0,1.0) );
			m_device->set_uniform( m_program, "light_specular", glm::vec4(1.0,1.0,1.0,1.0) );
		}

		{
			NV_PROFILE( "draw" );
			glm::mat4 model      = glm::mat4(1.0f);

			m_scene_state.set_model( model );
			m_legs->update( m_program );
			m_window->get_context()->draw( nv::TRIANGLES, m_render_state, m_scene_state, m_program, m_legs->get_mesh()->get_vertex_array(), m_legs->get_mesh()->get_index_count() );

			//model = m_legs->get_transform( "tag_torso", last_legs_frame, legs_frame, legs_interpolate );
			model = m_legs->get_transform( 0 );
			m_scene_state.set_model( model );
			m_torso->update( m_program );
			m_window->get_context()->draw( nv::TRIANGLES, m_render_state, m_scene_state, m_program, m_torso->get_mesh()->get_vertex_array(), m_torso->get_mesh()->get_index_count() );

			glm::mat4 head = model * m_torso->get_transform( 0 ); //, last_torso_frame, torso_frame, torso_interpolate );
			m_scene_state.set_model( head );
			m_head->update( m_program );
			m_window->get_context()->draw( nv::TRIANGLES, m_render_state, m_scene_state, m_program, m_head->get_mesh()->get_vertex_array(), m_head->get_mesh()->get_index_count() );

			glm::mat4 weapon = model * m_torso->get_transform( 2 ); //, last_torso_frame, torso_frame, torso_interpolate );
			m_weapon->update( m_program );
			m_scene_state.set_model( weapon );
			m_context->bind( m_diffuse_weapon, nv::TEX_DIFFUSE );
			m_context->draw( nv::TRIANGLES, m_render_state, m_scene_state, m_program, m_weapon->get_mesh()->get_vertex_array(), m_weapon->get_mesh()->get_index_count() );

		}

		{
			NV_PROFILE( "swap" );
			m_window->swap_buffers();
		}

		{
			NV_PROFILE( "pollevent" );
			nv::io_event event;
			while(m_window->poll_event(event)) 
			{      
				switch (event.type) 
				{
				case nv::EV_QUIT:
					keypress = 1;
					break;
				case nv::EV_KEY:
					if (event.key.pressed)
					{
						switch (event.key.code) 
						{
						case nv::KEY_ESCAPE : keypress = 1; break;
						case nv::KEY_1 : m_legs->setup_animation( 0, 89, 20, true ); m_torso->setup_animation( 0, 89, 20, true ); common = true; break;
						case nv::KEY_2 : m_legs->setup_animation( 90, 8, 20, true ); if (common) m_torso->setup_animation( 151, 1, 20, true ); common = false; break;
						case nv::KEY_3 : m_legs->setup_animation( 98, 12, 20, true ); if (common) m_torso->setup_animation( 151, 1, 20, true ); common = false; break;
						case nv::KEY_4 : m_legs->setup_animation( 110, 9, 15, true ); if (common) m_torso->setup_animation( 151, 1, 20, true ); common = false; break;
						case nv::KEY_5 : m_legs->setup_animation( 138, 10, 15, true ); if (common) m_torso->setup_animation( 151, 1, 20, true ); common = false; break;
						case nv::KEY_6 : m_legs->setup_animation( 171, 9, 15, true ); if (common) m_torso->setup_animation( 151, 1, 20, true ); common = false; break;
						case nv::KEY_Q : m_torso->setup_animation( 90, 40, 20, true ); if (common) m_legs->setup_animation( 171, 10, 15, true ); common = false; break;
						case nv::KEY_W : m_torso->setup_animation( 130, 6, 15, true ); if (common) m_legs->setup_animation( 171, 10, 15, true ); common = false; break;
						case nv::KEY_E : m_torso->setup_animation( 136, 6, 15, true ); if (common) m_legs->setup_animation( 171, 10, 15, true ); common = false; break;
						case nv::KEY_R : m_torso->setup_animation( 142, 5, 20, true ); if (common) m_legs->setup_animation( 171, 10, 15, true ); common = false; break;
						case nv::KEY_T : m_torso->setup_animation( 147, 4, 20, true ); if (common) m_legs->setup_animation( 171, 10, 15, true ); common = false; break;
						case nv::KEY_Y : m_torso->setup_animation( 151, 1, 15, true ); if (common) m_legs->setup_animation( 171, 10, 15, true ); common = false; break;
						case nv::KEY_U : m_torso->setup_animation( 152, 1, 15, true ); if (common) m_legs->setup_animation( 171, 10, 15, true ); common = false; break;
						case nv::KEY_F1 : nv::profiler::pointer()->log_report(); break;
						default: break;
						}
					}
					break;
				default: break;
				}
			}
		}
		fps_counter.tick();
	}
	return true;
}

application::~application()
{
	m_device->release( m_program );
	delete m_torso;
	delete m_legs;
	delete m_head;
	delete m_weapon;
	m_device->release( m_diffuse );
	m_device->release( m_diffuse_weapon );
	delete m_window;
	delete m_device;
}


int main(int, char* [])
{
	nv::logger log(nv::LOG_TRACE);
	log.add_sink( new nv::log_file_sink("log.txt"), nv::LOG_TRACE );
	log.add_sink( new nv::log_console_sink(), nv::LOG_TRACE );
	
	NV_LOG( nv::LOG_NOTICE, "Logging started" );
	application app;
	if ( app.initialize() )
	{
		app.run();
	}
	NV_LOG( nv::LOG_NOTICE, "Logging stopped" );

	return 0;
}

