// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file quaternion.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief quaternion ops
*/

#ifndef NV_STL_MATH_QUATERNION_HH
#define NV_STL_MATH_QUATERNION_HH

#include <nv/stl/math/common.hh>
#include <nv/stl/math/constants.hh>
#include <nv/stl/math/geometric.hh>
#include <nv/stl/math/exponential.hh>
#include <nv/stl/math/angle.hh>
#include <nv/base/cmath.hh>

namespace nv
{

	namespace math
	{

		template <typename T>
		struct tquat
		{
			typedef tquat<T> type;
			typedef T value_type;
		public:
			T x, y, z, w;

			typedef int length_type;

			constexpr length_type length() const { return 4; }
			inline T& operator[]( length_type i )
			{
				NV_ASSERT( i >= 0 && i < 4, "index out of range" );
				return ( &x )[i];
			}
			inline const T& operator[]( length_type i ) const
			{
				NV_ASSERT( i >= 0 && i < 4, "index out of range" );
				return ( &x )[i];
			}

			constexpr tquat() 
				: x( 0 ), y( 0 ), z( 0 ), w( 1 ) {}
			constexpr tquat( const tquat<T>& q ) 
				: x( q.x ), y( q.y ), z( q.z ), w( q.w ) {}
			
			inline explicit tquat( no_init_t ) {}
			constexpr explicit tquat( const T& s, const tvec3<T>& v )
				: x( v.x ), y( v.y ), z( v.z ), w( s ) {}
			constexpr tquat( const T& aw, const T& ax, const T& ay, const T& az )
				: x( ax ), y( ay ), z( az ), w( aw ) {}

			inline explicit operator tmat3<T>();
			inline explicit operator tmat4<T>();
			inline explicit tquat( const tvec3<T>& u, const tvec3<T>& v );
			inline explicit tquat( const tvec3<T>& euler );
			inline explicit tquat( const tmat3<T>& m );
			inline explicit tquat( const tmat4<T>& m );

			inline tquat<T>& operator=( const tquat<T>& q )
			{
				this->w = q.w;
				this->x = q.x;
				this->y = q.y;
				this->z = q.z;
				return *this;
			}
			inline tquat<T>& operator+=( const tquat<T>& q )
			{
				this->w += q.w;
				this->x += q.x;
				this->y += q.y;
				this->z += q.z;
				return *this;
			}
			inline tquat<T>& operator*=( const tquat<T>& q )
			{
				tquat<T> p( *this );
				
				this->w = p.w * q.w - p.x * q.x - p.y * q.y - p.z * q.z;
				this->x = p.w * q.x + p.x * q.w + p.y * q.z - p.z * q.y;
				this->y = p.w * q.y + p.y * q.w + p.z * q.x - p.x * q.z;
				this->z = p.w * q.z + p.z * q.w + p.x * q.y - p.y * q.x;
				return *this;
			}
			inline tquat<T>& operator*=( T s )
			{
				this->w *= s;
				this->x *= s;
				this->y *= s;
				this->z *= s;
				return *this;
			}

			inline tquat<T>& operator/=( T s )
			{
				this->w /= s;
				this->x /= s;
				this->y /= s;
				this->z /= s;
				return *this;
			}

		};

		namespace detail
		{

			template < typename T >
			struct dot_impl< tquat< T > >
			{
				inline static T get( const tquat<T>& a, const tquat<T>& b )
				{
					tvec4<T> tmp( a.x * b.x, a.y * b.y, a.z * b.z, a.w * b.w );
					return ( tmp.x + tmp.y ) + ( tmp.z + tmp.w );
				}
			};

		}

		template < typename T, typename enable_if< is_fp_quat<T>::value >::type* = nullptr >
		inline value_type_t<T> dot( const T& a, const T& b )
		{
			return detail::dot_impl< T >::get( a, b );
		}

		template < typename T >
		inline tquat<T> conjugate( const tquat<T>& q )
		{
			return tquat<T>( q.w, -q.x, -q.y, -q.z );
		}

		template < typename T >
		inline tquat<T> inverse( const tquat<T>& q )
		{
			return conjugate( q ) / dot( q, q );
		}

		template < typename T >
		inline T length( const tquat<T>& q )
		{
			return nv::sqrt( dot( q, q ) );
		}

		template < typename T >
		inline tquat<T> normalize( const tquat<T>& q )
		{
			T len = math::length( q );
			if ( len <= T( 0 ) ) 
				return tquat<T>( 1, 0, 0, 0 );
			T rlen = T( 1 ) / len;
			return tquat<T>( q.w * rlen, q.x * rlen, q.y * rlen, q.z * rlen );
		}

		template < typename T >
		inline tquat<T> cross( const tquat<T>& q1, const tquat<T>& q2 )
		{
			return tquat<T>(
				q1.w * q2.w - q1.x * q2.x - q1.y * q2.y - q1.z * q2.z,
				q1.w * q2.x + q1.x * q2.w + q1.y * q2.z - q1.z * q2.y,
				q1.w * q2.y + q1.y * q2.w + q1.z * q2.x - q1.x * q2.z,
				q1.w * q2.z + q1.z * q2.w + q1.x * q2.y - q1.y * q2.x );
		}

		template < typename T >
		inline tquat<T> mix( const tquat<T>& a, const tquat<T>& b, T m )
		{
			T cos_theta = dot( a, b );

			if ( cos_theta > T( 1 ) - epsilon<T>() )
			{
				return tquat<T>(
					mix( a.w, b.w, m ),
					mix( a.x, b.x, m ),
					mix( a.y, b.y, m ),
					mix( a.z, b.z, m ) );
			}
			else
			{
				T angle = nv::acos( cos_theta );
				return ( nv::sin( ( T( 1 ) - m ) * angle ) * a + nv::sin( m * angle ) * b ) / nv::sin( angle );
			}
		}

		template < typename T >
		inline tquat<T> lerp( const tquat<T>& a, const tquat<T>& b, T m )
		{
			tquat<T> aa = dot( a, b ) > T( 0 ) ? a : -a;
			return aa * ( T( 1 ) - m ) + ( b * m );
		}

		template < typename T >
		inline tquat<T> nlerp( const tquat<T>& a, const tquat<T>& b, T m )
		{
			tquat<T> aa = dot( a, b ) > T( 0 ) ? a : -a;
			return normalize( aa * ( T( 1 ) - m ) + ( b * m ) );
		}

		template < typename T >
		inline tquat<T> slerp( const tquat<T>& x, const tquat<T>& y, T m )
		{
			tquat<T> z = x;
			T cos_theta = dot( x, y );
			if ( cos_theta < T( 0 ) )
			{
				z = -x;
				cos_theta = -cos_theta;
			}
			if ( cos_theta > T( 1 ) - epsilon<T>() )
			{
				return tquat<T>(
					mix( z.w, y.w, m ),
					mix( z.x, y.x, m ),
					mix( z.y, y.y, m ),
					mix( z.z, y.z, m ) );
			}
			else
			{
				T angle = nv::acos( cos_theta );
				return ( nv::sin( ( T( 1 ) - m ) * angle ) * z + nv::sin( m * angle ) * y ) / nv::sin( angle );
			}
		}

		template < typename T >
		inline tquat<T> exp( const tquat<T>& q )
		{
			tvec3<T> u( q.x, q.y, q.z );
			T angle = math::length( u );
			if ( angle < epsilon<T>() )
				return tquat<T>();

			tvec3<T> v( u / angle );
			return tquat<T>( nv::cos( angle ), nv::sin( angle ) * v );
		}

		template < typename T >
		inline tquat<T> pow( const tquat<T>& x, const T& y )
		{
			if ( abs( x.w ) > ( static_cast<T>( 1 ) - epsilon<T>() ) )
				return x;
			T angle = acos( y );
			T nangle = angle * y;
			T div = sin( nangle ) / sin( angle );
			return normalize( tquat<T>(
				cos( nangle ),
				x.x * div,
				x.y * div,
				x.z * div ) );
		}
		template < typename T >
		inline tquat<T> log( const tquat<T>& q )
		{
			tvec3<T> u( q.x, q.y, q.z );
			T veclen = length( u );

			if ( veclen < epsilon<T>() )
			{
				if ( q.w > T( 0 ) )
					return tquat<T>( nv::log( q.w ), T( 0 ), T( 0 ), T( 0 ) );
				else if ( q.w < T( 0 ) )
					return tquat<T>( nv::log( -q.w ), pi<T>(), T( 0 ), T( 0 ) );
				else
					return tquat<T>( INFINITY, INFINITY, INFINITY, INFINITY );
			}
			else
			{
				T quatlen = nv::sqrt( veclen * veclen + q.w * q.w );
				T t = nv::atan2( veclen, T( q.w ) ) / veclen;
				return tquat<T>( nv::log( quatlen ), t * q.x, t * q.y, t * q.z );
			}
		}

		template < typename T >
		tquat<T> intermediate( const tquat<T>& prev, const tquat<T>& curr, const tquat<T>& next )
		{
			tquat<T> inv = inverse( curr );
			return curr * math::exp( ( math::log( inv * next ) + math::log( inv * prev ) ) / T( -4 ) );
		}


		template <typename T>
		inline tquat<T> squad( const tquat<T>& q1, const tquat<T>& q2, const tquat<T>& s1, const tquat<T>& s2, const T& h )
		{
			return slerp( slerp( q1, q2, h ), slerp( s1, s2, h ), T( 2 ) * ( T( 1 ) - h ) * h );
		}

		template < typename T >
		inline tquat<T> rotate( const tquat<T>& q, T angle, const tvec3<T>& v )
		{
			tvec3<T> temp = v;
			T len = math::length( temp );
			if ( abs( len - T( 1 ) ) > T( 0.001 ) )
			{
				T rlen = static_cast<T>( 1 ) / len;
				temp.x *= rlen;
				temp.y *= rlen;
				temp.z *= rlen;
			}

			T sina = sin( angle * T( 0.5 ) );
			return q * tquat<T>( cos( angle * T( 0.5 ) ), temp.x * sina, temp.y * sina, temp.z * sina );
		}

		template < typename T >
		inline tvec3<T> euler_angles( const tquat<T>& x )
		{
			return tvec3<T>( pitch( x ), yaw( x ), roll( x ) );
		}

		template < typename T >
		inline T roll( const tquat<T>& q )
		{
			return T( atan( T( 2 ) * ( q.x * q.y + q.w * q.z ), q.w * q.w + q.x * q.x - q.y * q.y - q.z * q.z ) );
		}

		template < typename T >
		inline T pitch( const tquat<T>& q )
		{
			return T( atan( T( 2 ) * ( q.y * q.z + q.w * q.x ), q.w * q.w - q.x * q.x - q.y * q.y + q.z * q.z ) );
		}

		template < typename T >
		inline T yaw( const tquat<T>& q )
		{
			return asin( T( -2 ) * ( q.x * q.z - q.w * q.y ) );
		}

		template < typename T >
		inline tmat3<T> mat3_cast( const tquat<T>& q )
		{
			tmat3<T> result( T( 1 ) );
			T qxx( q.x * q.x );
			T qyy( q.y * q.y );
			T qzz( q.z * q.z );
			T qxz( q.x * q.z );
			T qxy( q.x * q.y );
			T qyz( q.y * q.z );
			T qwx( q.w * q.x );
			T qwy( q.w * q.y );
			T qwz( q.w * q.z );

			result[0][0] = 1 - 2 * ( qyy + qzz );
			result[0][1] = 2 * ( qxy + qwz );
			result[0][2] = 2 * ( qxz - qwy );

			result[1][0] = 2 * ( qxy - qwz );
			result[1][1] = 1 - 2 * ( qxx + qzz );
			result[1][2] = 2 * ( qyz + qwx );

			result[2][0] = 2 * ( qxz + qwy );
			result[2][1] = 2 * ( qyz - qwx );
			result[2][2] = 1 - 2 * ( qxx + qyy );
			return result;
		}

		template < typename T >
		inline tmat4<T> mat4_cast( const tquat<T>& q )
		{
			return tmat4<T>( mat3_cast( q ) );
		}

		template < typename T >
		inline tquat<T> quat_cast( const tmat3<T>& m )
		{
			T fx = m[0][0] - m[1][1] - m[2][2];
			T fy = m[1][1] - m[0][0] - m[2][2];
			T fz = m[2][2] - m[0][0] - m[1][1];
			T fw = m[0][0] + m[1][1] + m[2][2];

			int imax = 0;
			T fmax = fw;
			if ( fx > fmax )
			{
				fmax = fx;
				imax = 1;
			}
			if ( fy > fmax )
			{
				fmax = fy;
				imax = 2;
			}
			if ( fz > fmax )
			{
				fmax = fz;
				imax = 3;
			}

			T vmax = nv::sqrt( fmax + T( 1 ) ) * T( 0.5 );
			T mult = static_cast<T>( 0.25 ) / vmax;

			tquat<T> result( no_init );
			switch ( imax )
			{
			case 0:
				result.w = vmax;
				result.x = ( m[1][2] - m[2][1] ) * mult;
				result.y = ( m[2][0] - m[0][2] ) * mult;
				result.z = ( m[0][1] - m[1][0] ) * mult;
				break;
			case 1:
				result.w = ( m[1][2] - m[2][1] ) * mult;
				result.x = vmax;
				result.y = ( m[0][1] + m[1][0] ) * mult;
				result.z = ( m[2][0] + m[0][2] ) * mult;
				break;
			case 2:
				result.w = ( m[2][0] - m[0][2] ) * mult;
				result.x = ( m[0][1] + m[1][0] ) * mult;
				result.y = vmax;
				result.z = ( m[1][2] + m[2][1] ) * mult;
				break;
			case 3:
				result.w = ( m[0][1] - m[1][0] ) * mult;
				result.x = ( m[2][0] + m[0][2] ) * mult;
				result.y = ( m[1][2] + m[2][1] ) * mult;
				result.z = vmax;
				break;

			default: break;
			}
			return result;
		}

		template < typename T >
		inline tquat<T> quat_cast( const tmat4<T>& m4 )
		{
			return quat_cast( tmat3<T>( m4 ) );
		}

		template < typename T >
		inline T angle( const tquat<T>& x )
		{
			return acos( x.w ) * T( 2 );
		}

		template < typename T >
		inline tvec3<T> axis( const tquat<T>& x )
		{
			T tmp1 = static_cast<T>( 1 ) - x.w * x.w;
			if ( tmp1 <= static_cast<T>( 0 ) ) return tvec3<T>( 0, 0, 1 );
			T tmp2 = static_cast<T>( 1 ) / sqrt( tmp1 );
			return tvec3<T>( x.x * tmp2, x.y * tmp2, x.z * tmp2 );
		}

		template < typename T >
		inline tquat<T> angle_axis( T angle, const tvec3<T>& v )
		{
			tquat<T> result( no_init );
			T sina = nv::sin( angle * static_cast<T>( 0.5 ) );
			result.w = nv::cos( angle * static_cast<T>( 0.5 ) );
			result.x = v.x * sina;
			result.y = v.y * sina;
			result.z = v.z * sina;
			return result;
		}

		template < typename T >
		inline tquat<T>::operator tmat3<T>()
		{
			return mat3_cast( *this );
		}

		template < typename T >
		inline tquat<T>::operator tmat4<T>()
		{
			return mat4_cast( *this );
		}

		template < typename T >
		inline tquat<T>::tquat( const tvec3<T>& u, const tvec3<T>& v )
		{
			const tvec3<T> lw( cross( u, v ) );
			T dot_result = detail::dot_impl< tvec3<T> >::get( u, v );
			tquat<T> q( T( 1 ) + dot_result, lw.x, lw.y, lw.z );
			*this = normalize( q );
		}

		template < typename T >
		inline tquat<T>::tquat( const tvec3<T>& euler )
		{
			tvec3<T> c = math::cos( euler * T( 0.5 ) );
			tvec3<T> s = math::sin( euler * T( 0.5 ) );

			w = c.x * c.y * c.z + s.x * s.y * s.z;
			x = s.x * c.y * c.z - c.x * s.y * s.z;
			y = c.x * s.y * c.z + s.x * c.y * s.z;
			z = c.x * c.y * s.z - s.x * s.y * c.z;
		}

		template < typename T >
		inline tquat<T>::tquat( const tmat3<T>& m )
		{
			*this = quat_cast( m );
		}

		template < typename T >
		inline tquat<T>::tquat( const tmat4<T>& m )
		{
			*this = quat_cast( m );
		}

		template < typename T >
		inline tquat<T> operator-( const tquat<T>& q )
		{
			return tquat<T>( -q.w, -q.x, -q.y, -q.z );
		}

		template < typename T >
		inline tquat<T> operator+( const tquat<T>& q, const tquat<T>& p )
		{
			return tquat<T>( q ) += p;
		}

		template < typename T >
		inline tquat<T> operator*( const tquat<T>& q, const tquat<T>& p )
		{
			return tquat<T>( q ) *= p;
		}

		template < typename T >
		inline tvec3<T> operator*( const tquat<T>& q, const tvec3<T>& v )
		{
			tvec3<T> qvec( q.x, q.y, q.z );
			tvec3<T> uv( math::cross( qvec, v ) );
			tvec3<T> uuv( math::cross( qvec, uv ) );

			return v + ( ( uv * q.w ) + uuv ) * static_cast<T>( 2 );
		}

		template < typename T >
		inline tvec3<T> operator*( const tvec3<T> & v, const tquat<T>& q )
		{
			return math::inverse( q ) * v;
		}

		template < typename T >
		inline tvec4<T> operator*( const tquat<T>& q, const tvec4<T>& v )
		{
			return tvec4<T>( q * tvec3<T>( v ), v.w );
		}

		template < typename T >
		inline tvec4<T> operator*( const tvec4<T>& v, const tquat<T>& q )
		{
			return math::inverse( q ) * v;
		}

		template < typename T >
		inline tquat<T> operator*( const tquat<T>& q, T s )
		{
			return tquat<T>( q.w * s, q.x * s, q.y * s, q.z * s );
		}

		template < typename T >
		inline tquat<T> operator*( T s, const tquat<T>& q )
		{
			return q * s;
		}

		template < typename T >
		inline tquat<T> operator/( const tquat<T>& q, T s )
		{
			return tquat<T>( q.w / s, q.x / s, q.y / s, q.z / s );
		}

		template < typename T >
		inline bool operator==( const tquat<T>& q1, const tquat<T>& q2 )
		{
			return ( q1.x == q2.x ) && ( q1.y == q2.y ) && ( q1.z == q2.z ) && ( q1.w == q2.w );
		}

		template < typename T >
		inline bool operator!=( const tquat<T>& q1, const tquat<T>& q2 )
		{
			return ( q1.x != q2.x ) || ( q1.y != q2.y ) || ( q1.z != q2.z ) || ( q1.w != q2.w );
		}

	}

}

#endif // NV_STL_MATH_QUATERNION_HH
