// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file context.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Context class
 */

#ifndef NV_INTERFACE_CONTEXT_HH
#define NV_INTERFACE_CONTEXT_HH

#include <nv/common.hh>
#include <nv/interface/device.hh>
#include <nv/interface/camera.hh>
#include <nv/interface/clear_state.hh>
#include <nv/interface/render_state.hh>

namespace nv
{
	struct vertex_array_tag {};
	struct framebuffer_tag {};
	typedef handle< uint32, 16, 16, vertex_array_tag > vertex_array;
	typedef handle< uint32, 16, 16, framebuffer_tag >  framebuffer;

	enum primitive
	{
		POINTS,
		LINES,
		LINE_LOOP,
		LINE_STRIP,
		TRIANGLES,
		TRIANGLE_STRIP,
		TRIANGLE_FAN
	};

	enum framebuffer_slot
	{
		READ_FRAMEBUFFER,
		DRAW_FRAMEBUFFER,
		FRAMEBUFFER,
	};

	struct vertex_array_info
	{
		static const int MAX_ATTRIBUTES = 16;

		uint32        count;
		buffer        index;
		bool          index_owner;
		datatype      index_type;
		vertex_buffer_attribute attr[MAX_ATTRIBUTES];
	};

	struct framebuffer_info
	{
		texture color_attachments[8];
		uint32  color_attachment_count;
		texture depth_attachment;
		uint32  sample_count;
	};

	class vertex_array_desc : private vertex_array_info
	{
	private:
		friend class context;
		friend class gl_context;
	public:
		vertex_array_desc()
		{
			count = 0;
			index = buffer();
			index_owner = false;
			index_type = USHORT;
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer( buffer vb, bool owned )
		{
			add_vertex_buffer_impl< VTX, SLOT >( vb, owned, has_slot< VTX, SLOT >() );
		}

		template < typename VTX >
		void add_vertex_buffers( buffer vb, bool owned )
		{
			add_vertex_buffer< VTX, slot::POSITION >( vb, owned );
			add_vertex_buffer< VTX, slot::TEXCOORD >( vb, owned );
			add_vertex_buffer< VTX, slot::NORMAL   >( vb, owned );
			add_vertex_buffer< VTX, slot::TANGENT >( vb, owned );
			add_vertex_buffer< VTX, slot::BONEINDEX >( vb, owned );
			add_vertex_buffer< VTX, slot::BONEWEIGHT >( vb, owned );
			add_vertex_buffer< VTX, slot::COLOR >( vb, owned );
		}

		void add_vertex_buffers( buffer buf, const data_descriptor& descriptor )
		{
			for ( const auto& cslot : descriptor )
			{
				const datatype_info& info = get_datatype_info( cslot.etype );
				add_vertex_buffer( cslot.vslot, buf, info.base, info.elements, cslot.offset, descriptor.element_size(), false );
			}
		}


		// TODO: should be private
		void add_vertex_buffer( slot location, buffer buf, datatype datatype, size_t components, size_t offset = 0, size_t stride = 0, bool owner = true )
		{
			NV_ASSERT( count < uint16( MAX_ATTRIBUTES ), "MAX_ATTRIBUTES reached!" );
			vertex_buffer_attribute& p = attr[count];
			p.vbuffer    = buf;
			p.dtype      = datatype;
			p.components = components;
			p.offset     = offset;
			p.stride     = stride;
			p.owner      = owner;
			p.location   = location;
			count++;
		}

		// TODO: should be private
		void set_index_buffer( buffer b, datatype datatype, bool owner )
		{
			index = b;
			index_owner = owner;
			index_type = datatype;
		}
	private:
		template < typename VTX, slot SLOT >
		void add_vertex_buffer_impl( buffer, bool, const false_type& )
		{
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer_impl( buffer vb, bool owned, const true_type& )
		{
			typedef slot_info< VTX, SLOT > vinfo;
			typedef datatype_traits< typename vinfo::value_type > dt_traits;
			add_vertex_buffer( SLOT, vb, type_to_enum< typename dt_traits::base_type >::type, dt_traits::size, vinfo::offset, sizeof( VTX ), owned );
		}

	};

	class context
	{
	public:
		context( device* a_device )	
		{ 
			m_device = a_device; 
		}
		virtual vertex_array create_vertex_array( const vertex_array_desc& desc ) = 0;
		virtual framebuffer create_framebuffer( uint32 temp_samples = 1 ) = 0;
		virtual void release( vertex_array ) = 0;
		virtual void release( framebuffer ) = 0;

		virtual void release( program p ) { m_device->release( p ); }
		virtual void release( texture t ) { m_device->release( t ); }
		virtual void release( buffer b ) { m_device->release( b ); }

		virtual texture create_texture( texture_type type, ivec2 size, image_format aformat, sampler asampler, const void* data = nullptr ) = 0;
		virtual texture create_texture( texture_type type, ivec3 size, image_format aformat, sampler asampler, const void* data = nullptr ) = 0;
		virtual texture create_texture( texture_type type, pixel_format format ) { return m_device->create_texture( type, format ); }
		// TODO: remove?
		virtual texture create_texture( ivec2 size, image_format aformat, sampler asampler, const void* data = nullptr )
		{ 
			return create_texture( TEXTURE_2D, size, aformat, asampler, data );
		}
		virtual texture create_texture( const image_data* data, sampler asampler )
		{
			return create_texture( data->get_size(), data->get_format(), asampler, data->get_data() );
		}


		virtual program create_program( string_view vs_source, string_view fs_source ) { return m_device->create_program( vs_source, fs_source ); }
		virtual buffer create_buffer( buffer_type type, buffer_hint hint, size_t size, const void* source = nullptr ) = 0;
		virtual void   create_buffer( buffer, size_t, const void* = nullptr ) = 0;


		virtual const framebuffer_info* get_framebuffer_info( framebuffer ) const = 0;

		virtual void set_draw_buffers( uint32 count, const output_slot* slots ) = 0;
		virtual void set_draw_buffer( output_slot slot ) = 0;
		virtual void set_read_buffer( output_slot slot ) = 0;
		virtual void blit( framebuffer f, buffer_mask mask, ivec2 src1, ivec2 src2, ivec2 dst1, ivec2 dst2 ) = 0;
		virtual void blit( framebuffer from, framebuffer to, buffer_mask mask, ivec2 src1, ivec2 src2, ivec2 dst1, ivec2 dst2 ) = 0;

		virtual void attach( framebuffer f, output_slot slot, texture t, int layer = -1 ) = 0;
		virtual void attach( framebuffer f, texture depth, int layer = -1 ) = 0;
		virtual void attach( framebuffer f, ivec2 size ) = 0;

		virtual bool check( framebuffer_slot ft ) = 0;
		virtual void bind( framebuffer f, framebuffer_slot slot = FRAMEBUFFER ) = 0;
		virtual void bind( texture, texture_slot ) = 0;
		virtual void bind( buffer, uint32 /*index*/, size_t /*offset*/ = 0, size_t /*size */= 0 ) = 0;
		virtual void bind( buffer, texture ) = 0;

		virtual void update( texture, const void* ) = 0;
		virtual void update( buffer, const void*, size_t /*offset*/, size_t /*size*/ ) = 0;
		virtual void* map_buffer( buffer, buffer_access, size_t /*offset*/, size_t /*length*/ ) = 0;
		virtual void unmap_buffer( buffer ) = 0;


//		virtual void update( buffer, uint32 /*index*/, const void* , size_t /*offset*/, size_t /*size*/ ) = 0;

		virtual void clear( const clear_state& cs ) = 0;
		// temporary
		virtual void draw( primitive prim, const render_state& rs, program p, vertex_array va, size_t count, size_t first = 0 ) = 0;
		virtual void draw_instanced( primitive prim, const render_state& rs, program p, size_t instances, vertex_array va, size_t count, size_t first = 0 ) = 0;

		void draw( primitive prim, const render_state& rs, const scene_state& s, program p, vertex_array va, size_t count, size_t first = 0 )
		{
			apply_engine_uniforms(p,s);
			draw( prim, rs, p, va, count, first );
		}
		virtual void apply_engine_uniforms( program p, const scene_state& s ) = 0;

		virtual void apply_render_state( const render_state& state ) = 0;
		virtual const ivec4& get_viewport() = 0;
		virtual void set_viewport( const ivec4& viewport ) = 0;
		virtual device* get_device() { return m_device; }
		virtual ~context() 
		{
		}

		template < typename VTXDATA >
		enable_if_t< is_class< VTXDATA >::value, vertex_array >
		create_vertex_array( const VTXDATA& data, buffer_hint hint )
		{
			return create_vertex_array( data.data(), data.size(), hint );
		}

		template < typename VTXDATA, typename IDXDATA >
		enable_if_t< is_class< VTXDATA >::value && is_class< IDXDATA >::value, vertex_array >
		create_vertex_array( const VTXDATA& data, const IDXDATA& idata, buffer_hint hint )
		{
			return create_vertex_array( data.data(), data.size(), idata.data(), idata.size(), hint );
		}

		template < typename VTX >
		vertex_array create_vertex_array( const VTX* v, size_t count, buffer_hint hint )
		{
			buffer       vb = create_buffer( VERTEX_BUFFER, hint, count * sizeof( VTX ), v );
			vertex_array_desc desc;
			desc.add_vertex_buffers< VTX >( vb, true );
			return create_vertex_array( desc );
		}

		template < typename VTX, typename IDX >
		vertex_array create_vertex_array( const VTX* v, size_t vcount, const IDX* i, size_t icount, buffer_hint hint )
		{
			buffer       vb = create_buffer( VERTEX_BUFFER, hint, count * sizeof( VTX ), v );
			buffer       ib = create_buffer( INDEX_BUFFER, hint, icount * sizeof( IDX ), i );
			vertex_array_desc desc;
			desc.add_vertex_buffers< VTX >( vb, true );
			desc.set_index_buffer( ib, type_to_enum< IDX >::type, true );
			return create_vertex_array( desc );
		}

		// TODO: HINTS ARE DIFFERENT!
		vertex_array create_vertex_array( const data_channel_set* data, buffer_hint hint )
		{
			vertex_array_desc va_desc;
			for ( auto& channel : *data )
			{
				if ( channel.size() > 0 )
				{
					const data_descriptor& desc = channel.descriptor();
					// TODO: no if switch
					if ( desc[0].vslot == slot::INDEX )
					{
						buffer b = create_buffer( INDEX_BUFFER, hint, channel.raw_size(), channel.raw_data() );
						va_desc.set_index_buffer( b, desc[0].etype, true );
					}
					else
					{
						buffer b = create_buffer( VERTEX_BUFFER, hint, channel.raw_size(), channel.raw_data() );
						va_desc.add_vertex_buffers( b, desc );
					}
				}
			}
			return create_vertex_array( va_desc );
		}

	protected:


		device*      m_device;
		clear_state  m_clear_state;
		render_state m_render_state;
		ivec4        m_viewport;
	};

} // namespace nv

#endif // NV_INTERFACE_CONTEXT_HH
