// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file gl_device.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Device class
 */

#ifndef NV_GL_DEVICE_HH
#define NV_GL_DEVICE_HH

#include <nv/interface/device.hh>

namespace nv
{
	struct gl_texture_info : public texture_info
	{
		unsigned glid;
	};

	struct gl_buffer_info : public buffer_info
	{
		unsigned glid;
	};

	struct gl_program_info : public program_info
	{
		unsigned glid;
		unsigned glidv;
		unsigned glidf;
		bool validated;
	};

	class gl_device : public device
	{
	public:
		friend class gl_context;

		gl_device();
		virtual image_data* create_image_data( string_view filename ); // temporary
		virtual image_data* create_image_data( const uint8* data, uint32 size ); // temporary

		virtual program create_program( string_view vs_source, string_view fs_source );
		virtual texture create_texture( texture_type type, pixel_format format );
		virtual buffer create_buffer( buffer_type type, buffer_hint hint );

		virtual void release( buffer b );
		virtual void release( texture t );
		virtual void release( program p );
		virtual const texture_info* get_texture_info( texture t ) const;
		virtual const buffer_info* get_buffer_info( buffer t ) const;

		virtual int get_attribute_location( program p, const string_view& name, bool fatal = true ) const;
		virtual bool bind_block( program p, const string_view& name, uint32 index );
		virtual int get_block_location( program p, const string_view& name, bool fatal = true ) const;
		virtual string_view get_shader_header() const { return m_shader_header; }
		virtual ~gl_device();
	protected:
		gl_texture_info* get_full_texture_info( texture t );
		gl_buffer_info* get_full_buffer_info( buffer t );

		void prepare_program( program p );
		uniform_base* get_uniform( program p, const string_view& name, bool fatal = true ) const;

	private:
		bool compile( gl_program_info* p, string_view vertex_program, string_view fragment_program );
		bool compile( uint32 sh_type, string_view shader_code, unsigned& glid );
		void update_uniforms( gl_program_info* p );
		void load_attributes( gl_program_info* p );
		void load_uniforms( gl_program_info* p );

		string_buffer m_shader_header;
		handle_store< gl_texture_info, texture >         m_textures;
		handle_store< gl_buffer_info,  buffer >          m_buffers;
		handle_store< gl_program_info, program >         m_programs;
	};

} // namespace nv


#endif // NV_GL_DEVICE_HH
