// Copyright (C) 2014 Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_CORE_TYPES_HH
#define NV_CORE_TYPES_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/memory.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/unordered_map.hh>
#include <nv/stl/string_table.hh>
#include <nv/stl/functional/hash.hh>
#include <nv/stl/rtti_types.hh>
#include <nv/stl/type_traits/properties.hh>

namespace nv
{

	struct type_hash_tag {};

	class thash64 : public hash_value< uint64, type_hash_tag >
	{
	public:
		typedef uint64 hash_type;
		typedef uint64 value_type;
		typedef thash64 this_type;
		typedef hash_value< uint64, type_hash_tag > inherited_type;

		// TODO: enforce exact type?
		constexpr thash64() : inherited_type() {}
		constexpr explicit thash64( hash_type value ) : inherited_type( value ) {}

		template < typename T >
		static thash64 create() { return thash64( rtti_type_hash< T >::hash() ); }
	};

	template< typename T >
	struct is_container
	{
	private:
		typedef char                      yes;
		typedef struct { char array[2]; } no;
		template<typename C> static yes test( typename C::iterator* );
		template<typename C> static no  test( ... );
	public:
		static const bool value = sizeof( test<T>( 0 ) ) == sizeof( yes );
	};

// 	template<>
// 	struct is_container < string_view >
// 	{
// 		static const bool value = false;
// 	};

	struct type_entry;
	class type_database;

	enum type_flag
	{
		TF_POINTER     = 0x01, //!< field is a pointer
		TF_NOSERIALIZE = 0x02, //!< ignore during serialization
		TF_INVISIBLE   = 0x04, //!< field invisible to API
		TF_READONLY    = 0x08, //!< read only field
		TF_SIMPLETYPE  = 0x10, //!< raw binary I/O possible
		TF_OWNED       = 0x20,
		TF_CONTAINER   = 0x40, //!< is a container
		TF_INDEXED     = 0x80, //!< type is a index to another type
	};

	struct type_field
	{
		shash64     name;     //!< name of the field
		type_entry* type;     //!< pointer to field type
		uint32      flags;    //!< flags 
		uint32      offset;   //!< offset into parent
		sint32      control;  //!< index to field control (in unions), -1 otherwise
		sint32      enumidx;  //!< field index (in unions)
	};

	struct type_enum
	{
		shash64 name;
		sint32  value;
	};

	struct type_entry
	{
		type_database*     type_db;     //!< Parent type database
		thash64            hash;        //!< type hash
		shash64            name;        //!< Scoped C++ name of the type
		constructor_t      constructor; //!< Pointers to the constructor 
		destructor_t       destructor;  //!< Pointers to the destructor 
		size_t             size;        //!< Result of sizeof(type) operation
		type_entry*        base_type;   //!< Base type
		vector<type_field> field_list;  //!< Field list
		vector<type_enum>  enum_list;   //!< Enum list
		hash_store< shash64, uint32 > field_names;
		hash_store< shash64, uint32 > enum_names;
	};

	class type_creator
	{
	public:
		type_creator( type_database* db, type_entry* entry )
			: m_database( db ), m_entry( entry ) {}

		template <typename TYPE>
		type_creator base();

		template< typename TOBJECT, typename TFIELD >
		type_creator field( const string_view& aname, TFIELD TOBJECT::*field, typename enable_if< is_container<TFIELD>::value, void* >::type = nullptr );

		template< typename TOBJECT, typename TFIELD >
		type_creator field( const string_view& aname, TFIELD TOBJECT::*field, typename enable_if< !is_container<TFIELD>::value, void* >::type = nullptr );

		template< typename TOBJECT, typename TFIELD, typename ENUM_VALUE >
		type_creator union_field( const string_view& aname, TFIELD TOBJECT::*field, const string_view& control_name, ENUM_VALUE control_value, typename enable_if< is_container<TFIELD>::value, void* >::type = nullptr );

		template< typename TOBJECT, typename TFIELD, typename ENUM_VALUE >
		type_creator union_field( const string_view& aname, TFIELD TOBJECT::*field, const string_view& control_name, ENUM_VALUE control_value, typename enable_if< !is_container<TFIELD>::value, void* >::type = nullptr );

		type_creator value( const string_view& aname, sint32 value );
	private:
		type_database* m_database;
		type_entry*    m_entry;
	};

	class type_database
	{
	public:
		friend class type_creator;

		template< typename TYPE >
		type_creator create_type()
		{
			return create_type<TYPE>( rtti_type_hash< TYPE >::name() );
		}

		template< typename TYPE >
		type_creator create_type( const string_view& name )
		{
			NV_ASSERT( !m_names.exists( name ), "Type redefinition!" );
			type_entry* i_type = new type_entry;
			i_type->type_db = this;
			i_type->hash = thash64::create< TYPE >();
			i_type->name = m_names.insert( name );
			i_type->size = sizeof( TYPE );

			i_type->constructor = raw_construct_object < TYPE >;
			i_type->destructor  = raw_destroy_object < TYPE >;
			m_index_by_type[ i_type->hash ] = i_type;
			m_index_by_name[ i_type->name ] = i_type;
			m_type_list.push_back( i_type );
			return type_creator( this, i_type );
		}

		type_entry* get_type( shash64 name )
		{
			auto it = m_index_by_name.find( name );
			return it != m_index_by_name.end() ? it->second : nullptr;
		}

		type_entry* get_type( thash64 hash )
		{
			auto it = m_index_by_type.find( hash );
			return it != m_index_by_type.end() ? it->second : nullptr;
		}

		template< typename T >
		type_entry* get_type()
		{
			auto it = m_index_by_type.find( thash64::create< T >() );
			return it != m_index_by_type.end() ? it->second : nullptr;
		}

		template< typename T >
		const type_entry* get_type() const
		{
			auto it = m_index_by_type.find( thash64::create< T >() );
			return it != m_index_by_type.end() ? it->second : nullptr;
		}

		string_view resolve_name( shash64 name_hash ) const
		{
			return m_names[name_hash];
		}

		string_view resolve_name( const type_entry* entry ) const
		{
			return m_names[entry->name];
		}

		~type_database()
		{
			for ( auto t : m_type_list ) delete t;
		}
	private:
		string_table                       m_names;
		vector<type_entry*>                m_type_list;
		hash_store< shash64, type_entry* > m_index_by_name;
		hash_store< thash64, type_entry* > m_index_by_type;
	};
	
	template < typename TYPE >
	type_creator type_creator::base()
	{
		m_entry->base_type = m_database->get_type< TYPE >();
		return *this;
	}

	template< typename TOBJECT, typename TFIELD >
	type_creator type_creator::field( const string_view& aname, TFIELD TOBJECT::*field, typename enable_if< is_container<TFIELD>::value, void* >::type )
	{
		typedef typename TFIELD::value_type field_type;
		NV_ASSERT( m_entry->enum_list.empty(), "Type cannot have both enums and fields!" );
		type_field f;
		f.name = m_database->m_names.insert( aname );
		f.type = m_database->get_type< remove_pointer_t<TFIELD> >();
		f.flags = TF_CONTAINER |
			( is_pointer<field_type>::value ? TF_POINTER : 0 ) |
			( is_pod<field_type>::value ? TF_SIMPLETYPE : 0 );
		f.offset = uint32( offset_of( field ) );
		f.control = -1;
		f.enumidx = 0;
		m_entry->field_list.push_back( f );
		m_entry->field_names[f.name] = m_entry->field_list.size() - 1;
		return *this;
	}

	template< typename TOBJECT, typename TFIELD >
	type_creator type_creator::field( const string_view& aname, TFIELD TOBJECT::*field, typename enable_if< !is_container<TFIELD>::value, void* >::type )
	{
		NV_ASSERT( m_entry->enum_list.empty(), "Type cannot have both enums and fields!" );
		type_field f;
		f.name = m_database->m_names.insert( aname );
		f.type = m_database->get_type< remove_pointer_t<TFIELD> >();
		f.flags =
			( is_pointer<TFIELD>::value ? TF_POINTER : 0 ) |
			( is_pod<TFIELD>::value ? TF_SIMPLETYPE : 0 );
		f.offset = uint32( offset_of( field ) );
		f.control = -1;
		f.enumidx = 0;
		m_entry->field_list.push_back( f );
		m_entry->field_names[f.name] = m_entry->field_list.size() - 1;
		return *this;
	}

	template< typename TOBJECT, typename TFIELD, typename ENUM_VALUE >
	type_creator type_creator::union_field( const string_view& aname, TFIELD TOBJECT::*field, const string_view& control_name, ENUM_VALUE control_value, typename enable_if< is_container<TFIELD>::value, void* >::type )
	{
		typedef typename TFIELD::value_type field_type;
		NV_ASSERT( m_entry->enum_list.empty(), "Type cannot have both enums and fields!" );
		type_field f;
		f.name = m_database->m_names.insert( aname );
		f.type = m_database->get_type< remove_pointer_t<TFIELD> >();
		f.flags = TF_CONTAINER |
			( is_pointer<field_type>::value ? TF_POINTER : 0 ) |
			( is_pod<field_type>::value ? TF_SIMPLETYPE : 0 );
		f.offset = uint32( offset_of( field ) );
		f.control = sint32( m_entry->field_names[ control_name ] );
		f.enumidx = static_cast< sint32 >( control_value );
		m_entry->field_list.push_back( f );
		m_entry->field_names[f.name] = m_entry->field_list.size() - 1;
		return *this;
	}

	template< typename TOBJECT, typename TFIELD, typename ENUM_VALUE >
	type_creator type_creator::union_field( const string_view& aname, TFIELD TOBJECT::*field, const string_view& control_name, ENUM_VALUE control_value, typename enable_if< !is_container<TFIELD>::value, void* >::type )
	{
		NV_ASSERT( m_entry->enum_list.empty(), "Type cannot have both enums and fields!" );
		type_field f;
		f.name = m_database->m_names.insert( aname );
		f.type = m_database->get_type< remove_pointer_t<TFIELD> >();
		f.flags =
			( is_pointer<TFIELD>::value ? TF_POINTER : 0 ) |
			( is_pod<TFIELD>::value ? TF_SIMPLETYPE : 0 );
		f.offset = uint32( offset_of( field ) );
		f.control = sint32( m_entry->field_names[control_name] );
		f.enumidx = static_cast<sint32>( control_value );
		m_entry->field_list.push_back( f );
		m_entry->field_names[f.name] = m_entry->field_list.size() - 1;
		return *this;
	}

	inline type_creator type_creator::value( const string_view& aname, sint32 value )
	{
		NV_ASSERT( m_entry->field_list.empty(), "Type cannot have both enums and fields!" );
		type_enum e;
		e.name = m_database->m_names.insert( aname );
		e.value = value;
		m_entry->enum_list.push_back( e );
		m_entry->enum_names[e.name] = m_entry->enum_list.size() - 1;
		return *this;
	}

	void register_core_types( type_database* db );

}


#endif // NV_CORE_TYPES_HH
