// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_CORE_RANDOM_HH
#define NV_CORE_RANDOM_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/type_traits/primary.hh>

namespace nv
{

	class random_base
	{
	public:
		typedef uint32 result_type; 
		typedef uint32 seed_type;

		virtual seed_type set_seed( seed_type seed = 0 ) = 0;
		virtual result_type rand() = 0;

		seed_type randomize()
		{
			return set_seed( randomized_seed() );
		}

		uint32 urand( uint32 val )
		{
			uint32 x, max = 0xFFFFFFFFUL - ( 0xFFFFFFFFUL % val );
			while ( ( x = rand() ) >= max );
			return x / ( max / val );
		}

		sint32 srand( sint32 val )
		{
			NV_ASSERT( val >= 0, "Bad srand range!" );
			return static_cast<sint32>( urand( static_cast<uint32>( val ) ) );
		}

		f32 frand()
		{
			return rand() / 4294967296.0f;
		}

		f32 frand( f32 val )
		{
			return frand() * val;
		}

		inline sint32 srange( sint32 min, sint32 max )
		{
			NV_ASSERT( max >= min, "Bad srange argument!" );
			// this method probably reduces range //
			uint32 roll = urand( static_cast<uint32>( max - min ) + 1 );
			return static_cast<sint32>( roll ) + min;
		}

		inline uint32 urange( uint32 min, uint32 max )
		{
			NV_ASSERT( max >= min, "Bad srange argument!" );
			return urand( max - min + 1 ) + min;
		}

		inline f32 frange( f32 min, f32 max )
		{
			NV_ASSERT( max >= min, "Bad srange argument!" );
			return frand( max - min ) + min;
		}

		uint32 dice( uint32 count, uint32 sides )
		{
			if ( count == 0 || sides == 0 ) return 0;
			uint32 result = count;
			while ( count-- > 0 )
			{
				result += urand( sides );
			};
			return result;
		}

		template < typename T >
		math::tvec2<T> range( math::tvec2<T> min, math::tvec2<T> max )
		{
			return math::tvec2<T>(
				range_impl( min.x, max.x, is_floating_point<T>() ),
				range_impl( min.y, max.y, is_floating_point<T>() )
				);
		}

		template < typename T >
		math::tvec3<T> range( math::tvec3<T> min, math::tvec3<T> max )
		{
			return math::tvec3<T>(
				range_impl( min.x, max.x, is_floating_point<T>() ),
				range_impl( min.y, max.y, is_floating_point<T>() ),
				range_impl( min.z, max.z, is_floating_point<T>() )
				);
		}

		template < typename T >
		math::tvec4<T> range( math::tvec4<T> min, math::tvec4<T> max )
		{
			return math::tvec4<T>(
				range_impl( min.x, max.x, is_floating_point<T>() ),
				range_impl( min.y, max.y, is_floating_point<T>() ),
				range_impl( min.z, max.z, is_floating_point<T>() ),
				range_impl( min.w, max.w, is_floating_point<T>() )
				);
		}



		vec3 unit_vec3( bool = false )
		{
			return precise_unit_vec3();
			//			return precise ? precise_unit_vec3() : fast_unit_vec3();
		}
		vec2 unit_vec2( bool = false )
		{
			return precise_unit_vec2();
			//			return precise ? precise_unit_vec2() : fast_unit_vec2();
		}

		vec2 disk_point( bool precise = false )
		{
			return precise ? precise_disk_point() : fast_disk_point();
		}

		vec3 sphere_point( bool precise = false )
		{
			return precise ? precise_sphere_point() : fast_sphere_point();
		}

		vec2 ellipse_point( const vec2& radii, bool precise = false )
		{
			return precise ? precise_ellipse_point( radii ) : fast_ellipse_point( radii );
		}

		vec3 ellipsoid_point( const vec3& radii, bool precise = false )
		{
			return precise ? precise_ellipsoid_point( radii ) : fast_ellipsoid_point( radii );
		}

		vec2 ellipse_edge( const vec2& radii, bool = false )
		{
			return unit_vec2() * radii;
		}

		vec3 ellipsoid_edge( const vec3& radii, bool = false )
		{
			return unit_vec3() * radii;
		}

		vec2 hollow_disk_point( float iradius, float oradius, bool precise = false )
		{
			return precise ? precise_hollow_disk_point( iradius, oradius ) : fast_hollow_disk_point( iradius, oradius );
		}

		vec3 hollow_sphere_point( float iradius, float oradius, bool precise = false )
		{
			return precise ? precise_hollow_sphere_point( iradius, oradius ) : fast_hollow_sphere_point( iradius, oradius );
		}

		vec2 hollow_ellipse_point( const vec2& iradii, const vec2& oradii, bool precise = false )
		{
			return precise ? precise_hollow_ellipse_point( iradii, oradii ) : fast_hollow_ellipse_point( iradii, oradii );
		}

		vec3 hollow_ellipsoid_point( const vec3& iradii, const vec3& oradii, bool precise = false )
		{
			return precise ? precise_hollow_ellipsoid_point( iradii, oradii ) : fast_hollow_ellipsoid_point( iradii, oradii );
		}

		//vec2 fast_unit_vec2();
		vec2 precise_unit_vec2();
		//vec3 fast_unit_vec3();
		vec3 precise_unit_vec3();

		vec2 fast_disk_point();
		vec2 precise_disk_point();
		vec3 fast_sphere_point();
		vec3 precise_sphere_point();

		vec2 fast_ellipse_point( const vec2& radii )
		{
			return fast_disk_point() * radii;
		}
		vec2 precise_ellipse_point( const vec2& radii );

		vec3 fast_ellipsoid_point( const vec3& radii )
		{
			return fast_sphere_point() * radii;
		}

		vec3 precise_ellipsoid_point( const vec3& radii );

		vec2 fast_hollow_disk_point( float iradius, float oradius );
		vec2 precise_hollow_disk_point( float iradius, float oradius );
		vec3 fast_hollow_sphere_point( float iradius, float oradius );
		vec3 precise_hollow_sphere_point( float iradius, float oradius );

		vec2 fast_hollow_ellipse_point( const vec2& iradii, const vec2& oradii );
		vec2 precise_hollow_ellipse_point( const vec2& iradii, const vec2& oradii );
		vec3 fast_hollow_ellipsoid_point( const vec3& iradii, const vec3& oradii );
		vec3 precise_hollow_ellipsoid_point( const vec3& iradii, const vec3& oradii );

	protected:
		static seed_type randomized_seed();

		template <typename T>
		T range_impl( T min, T max, const true_type& )
		{
			return frange( min, max );
		}

		template <typename T>
		T range_impl( T min, T max, const false_type& )
		{
			return srange( min, max );
		}
	};

	class random_mersenne : public random_base
	{
	public:
		static constexpr uint32 mersenne_n = 624;
		static constexpr uint32 mersenne_m = 397;
		static constexpr uint32 mersenne_static_seed = 5489;

		explicit random_mersenne( seed_type seed = randomized_seed() );
		virtual seed_type set_seed( seed_type seed = 0 );
		virtual result_type rand();
	private:
		void mt_init( uint32 seed );
		void mt_update();
		uint32 mt_uint32();

		uint32  m_state[mersenne_n];
		uint32* m_next;
		uint32  m_remaining;
		uint32  m_seeded : 1;
		uint32  m_static_system_seed : 1;
	};

	class random_xor128 : public random_base
	{
	public:
		explicit random_xor128( seed_type seed = randomized_seed() );
		virtual seed_type set_seed( seed_type seed = 0 );
		virtual result_type rand();
	private:
		uint32  m_state[4]; // xyzw
	};


	class random : public random_mersenne
	{
	public:
		explicit random( seed_type seed = randomized_seed() ) : random_mersenne( seed ) {};
		static random& get();
	};

}

#endif // NV_CORE_RANDOM_HH
