// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_ENGINE_PARTICLE_ENGINE
#define NV_ENGINE_PARTICLE_ENGINE

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/unordered_map.hh>
#include <nv/stl/handle.hh>
#include <nv/lua/lua_state.hh>
#include <nv/gfx/texture_atlas.hh>
#include <nv/interface/scene_node.hh>
#include <nv/interface/context.hh>

namespace nv
{
	static const unsigned MAX_PARTICLE_EMMITERS  = 8;
	static const unsigned MAX_PARTICLE_AFFECTORS = 8;

	struct particle_emmiter_data;
	struct particle_affector_data;
	struct particle;

	typedef void (*particle_emmiter_func)( const particle_emmiter_data*, particle*, uint32 count );
	typedef void (*particle_affector_func)( const particle_affector_data*, particle*, float factor, uint32 count );
	typedef bool (*particle_affector_init_func)( lua::table_guard* table, particle_affector_data* data );

	struct particle_affector_funcs
	{
		particle_affector_func      process;
		particle_affector_init_func init;
	};

	enum class particle_orientation
	{
		POINT,
		ORIENTED,
		ORIENTED_COMMON,
		PERPENDICULAR,
		PERPENDICULAR_COMMON,
	};
	enum class particle_origin
	{
		CENTER,
		TOP_LEFT,
		TOP_CENTER,
		TOP_RIGHT,
		CENTER_LEFT,
		CENTER_RIGHT,
		BOTTOM_LEFT,
		BOTTOM_CENTER,
		BOTTOM_RIGHT,
	};

	struct particle_vtx
	{
		vec3 position;
		vec2 texcoord;
		vec4 color;
	};

	struct particle_quad
	{
		particle_vtx data[6];
	};

	struct particle
	{
		vec3   position;
		vec4   color;
		vec3   velocity;
		vec2   size;
		vec2   tcoord_a;
		vec2   tcoord_b;
		float  rotation;
		float  lifetime;
		float  death;
	};

	struct particle_affector_data
	{
		particle_affector_func process;
		uint8 paramters[4*8*16];
	};

	struct particle_emmiter_data
	{
		particle_emmiter_func emmiter_func;
		vec3   position;
		vec3   extents;
		vec3   hextents;
		vec3   iextents;
		vec3   ihextents;
		vec4   color_min;
		vec4   color_max;
		vec2   size_min;
		vec2   size_max;
		bool   precise;
		bool   square;
		float  angle;
		float  velocity_min;
		float  velocity_max;
		float  lifetime_min;
		float  lifetime_max;
		vec3   dir;
		vec3   odir;
		vec3   cdir;
		float  duration_min;
		float  duration_max;
		float  repeat_min;
		float  repeat_max;
		float  rate;
	};

	struct particle_emmiter_info
	{
		bool   active;
		float  pause;
		float  next;
	};

	struct particle_system_data
	{
		uint32 quota;
		bool   local;
		vec3   common_up;
		vec3   common_dir;
		bool   accurate_facing;
		particle_orientation   orientation;
		particle_origin        origin;
		uint32                 emmiter_count;
		particle_emmiter_data  emmiters[MAX_PARTICLE_EMMITERS]; 
		uint32                 affector_count;
		particle_affector_data affectors[MAX_PARTICLE_AFFECTORS]; 
	};

	struct particle_system_group_tag {};
	typedef handle< uint32, 16, 16, particle_system_group_tag > particle_system_group;
	struct particle_system_tag {};
	typedef handle< uint32, 16, 16, particle_system_tag > particle_system;

	struct particle_system_info
	{
		particle*             particles;
		particle_emmiter_info emmiters[MAX_PARTICLE_EMMITERS];

		uint32                count;
		vec2                  texcoords[2];
		particle_system_group group;

		const particle_system_data*    data;
	};

	struct particle_system_group_info
	{
		uint32                count;
		uint32                quota;
		vertex_array          vtx_array;
		buffer                vtx_buffer;
		bool                  local;
		particle_quad*        quads;
		uint32                ref_counter;
	};

	struct particle_render_data
	{
		uint32                count;
		vertex_array          vtx_array;
	};

	class particle_engine
	{
	public:
		particle_engine( context* a_context );
		void reset();
		bool loaded( const string_view& system_id );
		void load( lua::table_guard& table );
		void draw( particle_system_group group, const render_state& rs, const scene_state& ss );
		void prepare( particle_system_group group );
		particle_system_group create_group( uint32 max_particles );
		particle_system create_system( const string_view& id, particle_system_group group );
		void release( particle_system_group group );
		void release( particle_system system );
		void update( particle_system system, float dtime );
		void render( particle_system system, const scene_state& s );
		void set_texcoords( particle_system system, vec2 a, vec2 b );
		void register_emmiter_type( const string_view& name, particle_emmiter_func func );
		void register_affector_type( const string_view& name, particle_affector_init_func init, particle_affector_func process );
		particle_render_data get_render_data( particle_system_group group );
		~particle_engine();
	private:
		void clear();
		void register_standard_emmiters();
		void register_standard_affectors();
		void generate_data( particle_system_info* info, const scene_state& s );
		void destroy_particles( particle_system_info* info, float dtime );
		void create_particles( particle_system_info* info, float dtime );
		void update_particles( particle_system_info* info, float dtime );
		void update_emmiters( particle_system_info* info, float dtime );

		device*  m_device;
		context* m_context;

		program  m_program_local;
		program  m_program_world;

		handle_store< particle_system_info, particle_system >             m_systems;
		handle_store< particle_system_group_info, particle_system_group > m_groups;

		hash_store< shash64, uint32 >                         m_names;
		vector< particle_system_data >                        m_data; 
		hash_store< shash64, particle_emmiter_func >          m_emmiters;
		hash_store< shash64, particle_affector_funcs >        m_affectors;
	};

}

#endif // NV_ENGINE_PARTICLE_ENGINE
