// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file gl_context.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Context class
 */

#ifndef NV_GL_CONTEXT_HH
#define NV_GL_CONTEXT_HH

#include <nv/interface/context.hh>

namespace nv
{
	struct gl_framebuffer_info : public framebuffer_info
	{
		unsigned glid;
		unsigned depth_rb_glid;
	};

	struct gl_vertex_array_info : public vertex_array_info
	{
		unsigned glid;
	};


	class gl_context : public context
	{
	public:
		gl_context( device* a_device, void* a_handle );
		virtual ~gl_context();

		virtual vertex_array create_vertex_array( const vertex_array_desc& desc );
		virtual framebuffer create_framebuffer( uint32 temp_samples = 1 );
		virtual void release( vertex_array va );
		virtual void release( framebuffer f );
		virtual const framebuffer_info* get_framebuffer_info( framebuffer f ) const;

		virtual void set_draw_buffers( uint32 count, const output_slot* slots );
		virtual void set_draw_buffer( output_slot slot );
		virtual void set_read_buffer( output_slot slot );
		virtual void blit( framebuffer f, clear_state::buffers_type mask, ivec2 src1, ivec2 src2, ivec2 dst1, ivec2 dst2 );
		virtual void blit( framebuffer from, framebuffer to, clear_state::buffers_type mask, ivec2 src1, ivec2 src2, ivec2 dst1, ivec2 dst2 );

		virtual void attach( framebuffer f, output_slot slot, texture t );
		virtual void attach( framebuffer f, texture depth, int layer = -1 );
		virtual void attach( framebuffer f, ivec2 size );
		virtual bool check( framebuffer_slot ft );
		virtual void bind( framebuffer f, framebuffer_slot ft = FRAMEBUFFER );
		virtual void bind( texture t, texture_slot slot );
		virtual void bind( buffer b, uint32 index, size_t offset = 0, size_t size = 0 );
		virtual void bind( buffer b, texture t );


		virtual void update( texture t, const void* data );
		virtual void update( buffer b, const void* data, size_t offset, size_t size );
		virtual void* map_buffer( buffer, buffer_access, size_t /*offset*/, size_t /*length*/ );
		virtual void unmap_buffer( buffer );

		//		virtual void update( buffer b, uint32 index, const void* data, size_t offset, size_t size );

		virtual void clear( const clear_state& cs );
		// temporary
		virtual void draw( primitive prim, const render_state& rs, program p, vertex_array va, size_t count, size_t first = 0 );
		virtual const ivec4& get_viewport();
		virtual void set_viewport( const ivec4& viewport );
		virtual void apply_render_state( const render_state& state );
		virtual void apply_engine_uniforms( program p, const scene_state& s );
		// TODO: remove
		void* get_native_handle() { return m_handle; }

	protected:
		void bind( program p );
//		void bind( buffer b );
		void bind( vertex_array va );
		void unbind( program p );
//		void unbind( buffer b );
		void unbind( vertex_array va );
		void unbind( framebuffer va );

		void force_apply_render_state( const render_state& state );
		void force_apply_stencil_face( unsigned face, const stencil_test_face& stencil );
	private:
		void apply_stencil_test( const stencil_test& stencil );
		void apply_stencil_face( unsigned face, stencil_test_face& stencil, const stencil_test_face& new_stencil );
		void apply_scissor_test( const scissor_test& scissor );
		void apply_depth_test( const depth_test& depth );
		void apply_depth_mask( bool mask );
		void apply_multisample( bool multisample );
		void apply_depth_range( const depth_range& range );
		void apply_color_mask( const color_mask& mask );
		void apply_blending( const blending& blend );
		void apply_culling( const culling& cull );
		void apply_polygon_mode( const polygon_mode& mode );
		void enable( unsigned int what, bool value );
		void set_active_texture( texture_slot slot );

		bool validate_program( program p );
	private:
		texture_slot m_active_slot;
		texture      m_bound_textures[ texture_slot::MAX_TEXTURES ];
		vec4  m_clear_color;
		float m_clear_depth;
		int   m_clear_stencil;
		void* m_handle;

		handle_store< gl_vertex_array_info, vertex_array > m_vertex_arrays;
		handle_store< gl_framebuffer_info, framebuffer >   m_framebuffers;
	};


} // namespace nv

#endif // NV_CONTEXT_HH
