// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file mat2.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief matrix 2x2
*/

#ifndef NV_STL_MATH_MAT2_HH
#define NV_STL_MATH_MAT2_HH

#include <nv/stl/math/common.hh>

namespace nv
{

	namespace math
	{

		template < typename T >
		struct tmat2
		{
			typedef tvec2<T> column_type;
			typedef tvec2<T> row_type;
			typedef tmat2<T> this_type;
			typedef size_t   size_type;
			typedef T        value_type;

			static constexpr size_type SIZE = 2;
			inline constexpr size_type size() const { return 2; }

		private:
			column_type value[2];

		public:

			inline tmat2()
				: value{
					column_type( static_cast<T>( 1 ), static_cast<T>( 0 ) ),
					column_type( static_cast<T>( 0 ), static_cast<T>( 1 ) )
				}
			{}

			inline tmat2( const tmat2<T> & m ) = default;
			inline tmat2( tmat2<T> && m ) = default;
			inline tmat2<T> & operator=( const tmat2<T> & m ) = default;
			inline tmat2<T> & operator=( tmat2<T> && m ) = default;

			inline explicit tmat2( T x )
				: value{
					column_type( static_cast<T>( x ), static_cast<T>( 0 ) ),
					column_type( static_cast<T>( 0 ), static_cast<T>( x ) )
				}
			{}

			inline explicit tmat2( no_init_t ) 
				: value{ 
					column_type( no_init ),
					column_type( no_init )
				}
			{}

			inline tmat2( 
				T x0, T y0,
				T x1, T y1
			) : value{
				column_type( x0, y0 ),
				column_type( x1, y1 )
			}
			{}
			
			inline tmat2(
				const column_type & v0,
				const column_type & v1
			) : value{ v0, v1 }
			{}

			template <typename U>
			inline explicit tmat2( const tmat2<U> & m ) 
				: value{ 
					column_type( m[0] ), 
					column_type( m[1] )
				}
			{}

			inline explicit tmat2( const tmat3<T> & m )
				: value{ 
					column_type( m[0] ), 
					column_type( m[1] )
				}
			{}

			inline explicit tmat2( const tmat4<T> & m )
				: value{ 
					column_type( m[0] ), 
					column_type( m[1] )
				}
			{}

			inline column_type & operator[]( size_type i )
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return this->value[i];
			}

			inline const column_type& operator[]( size_type i ) const
			{
				NV_ASSERT_DEBUG( i >= 0 && i < this->size(), "index out of range!" );
				return this->value[i];
			}


			inline tmat2<T> & operator+=( T s )
			{
				this->value[0] += s;
				this->value[1] += s;
				return *this;
			}

			inline tmat2<T> & operator+=( const tmat2<T> & m )
			{
				this->value[0] += m[0];
				this->value[1] += m[1];
				return *this;
			}

			inline tmat2<T> & operator-=( T s )
			{
				this->value[0] -= s;
				this->value[1] -= s;
				return *this;
			}

			inline tmat2<T> & operator-=( const tmat2<T> & m )
			{
				this->value[0] -= m[0];
				this->value[1] -= m[1];
				return *this;
			}

			inline tmat2<T> & operator*=( T s )
			{
				this->value[0] *= s;
				this->value[1] *= s;
				return *this;
			}

			inline tmat2<T> & operator*=( const tmat2<T> & m )
			{
				return ( *this = *this * m );
			}

			inline tmat2<T> & operator/=( T s )
			{
				this->value[0] /= s;
				this->value[1] /= s;
				return *this;
			}

			inline tmat2<T> & operator/=( const tmat2<T> & m )
			{
				return ( *this = *this * detail::compute_inverse<T>( m ) );
			}

			inline tmat2<T> & operator++ ()
			{
				++this->value[0];
				++this->value[1];
				return *this;
			}

			inline tmat2<T> & operator-- ()
			{
				--this->value[0];
				--this->value[1];
				return *this;
			}

			inline tmat2<T> operator++( int )
			{
				tmat2<T> result( *this );
				++*this;
				return result;
			}

			inline tmat2<T> operator--( int )
			{
				tmat2<T> result( *this );
				--*this;
				return result;
			}

		};

		namespace detail
		{
			template <typename T>
			inline tmat2<T> compute_inverse( const tmat2<T> & m )
			{
				T one_over_det = static_cast<T>( 1 ) / ( m[0][0] * m[1][1] - m[1][0] * m[0][1] );

				return tmat2<T>(
					+m[1][1] * one_over_det,
					-m[0][1] * one_over_det,
					-m[1][0] * one_over_det,
					+m[0][0] * one_over_det
					);
			}

		}

		template <typename T>
		inline tmat2<T> const operator-( const tmat2<T> & m )
		{
			return tmat2<T>(
				-m[0],
				-m[1] );
		}

		template <typename T>
		inline tmat2<T> operator+( const tmat2<T> & m, T s )
		{
			return tmat2<T>(
				m[0] + s,
				m[1] + s );
		}

		template <typename T>
		inline tmat2<T> operator+( T s, const tmat2<T> & m )
		{
			return tmat2<T>(
				m[0] + s,
				m[1] + s );
		}

		template <typename T>
		inline tmat2<T> operator+( const tmat2<T> & m1, const tmat2<T> & m2 )
		{
			return tmat2<T>(
				m1[0] + m2[0],
				m1[1] + m2[1] );
		}

		template <typename T>
		inline tmat2<T> operator-( const tmat2<T> & m, T s )
		{
			return tmat2<T>(
				m[0] - s,
				m[1] - s );
		}

		template <typename T>
		inline tmat2<T> operator-( T s, const tmat2<T> & m )
		{
			return tmat2<T>(
				s - m[0],
				s - m[1] );
		}

		template <typename T>
		inline tmat2<T> operator-( const tmat2<T> & m1, const tmat2<T> & m2 )
		{
			return tmat2<T>(
				m1[0] - m2[0],
				m1[1] - m2[1] );
		}

		template <typename T>
		inline tmat2<T> operator*( const tmat2<T> & m, T s )
		{
			return tmat2<T>(
				m[0] * s,
				m[1] * s );
		}

		template <typename T>
		inline tmat2<T> operator*( T s, const tmat2<T> & m )
		{
			return tmat2<T>(
				m[0] * s,
				m[1] * s );
		}

		template <typename T>
		inline typename tmat2<T>::column_type operator* ( const tmat2<T> & m, const typename tmat2<T>::row_type & v )
		{
			return tvec2<T>(
				m[0][0] * v.x + m[1][0] * v.y,
				m[0][1] * v.x + m[1][1] * v.y );
		}

		template <typename T>
		inline typename tmat2<T>::row_type operator* ( const typename tmat2<T>::column_type & v, const tmat2<T> & m )
		{
			return tvec2<T>(
				v.x * m[0][0] + v.y * m[0][1],
				v.x * m[1][0] + v.y * m[1][1] );
		}

		template <typename T>
		inline tmat2<T> operator*( const tmat2<T> & m1, const tmat2<T> & m2 )
		{
			return tmat2<T>(
				m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
				m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
				m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
				m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] );
		}

		template <typename T>
		inline tmat2<T> operator/( const tmat2<T> & m, T s )
		{
			return tmat2<T>(
				m[0] / s,
				m[1] / s );
		}

		template <typename T>
		inline tmat2<T> operator/( T s, const tmat2<T> & m )
		{
			return tmat2<T>(
				s / m[0],
				s / m[1] );
		}

		template <typename T>
		inline typename tmat2<T>::column_type operator/( const tmat2<T> & m, const typename tmat2<T>::row_type & v )
		{
			return detail::compute_inverse<T>( m ) * v;
		}

		template <typename T>
		inline typename tmat2<T>::row_type operator/( const typename tmat2<T>::column_type & v, const tmat2<T> & m )
		{
			return v * detail::compute_inverse<T>( m );
		}

		template <typename T>
		inline tmat2<T> operator/( const tmat2<T> & m1, const tmat2<T> & m2 )
		{
			tmat2<T> m1_copy( m1 );
			return m1_copy /= m2;
		}

		template <typename T>
		inline bool operator==( const tmat2<T> & m1, const tmat2<T> & m2 )
		{
			return ( m1[0] == m2[0] ) && ( m1[1] == m2[1] );
		}

		template <typename T>
		inline bool operator!=( const tmat2<T> & m1, const tmat2<T> & m2 )
		{
			return ( m1[0] != m2[0] ) || ( m1[1] != m2[1] );
		}

	}

}

#endif // NV_STL_MATH_MAT2_HH
