#include <nv/gfx/keyframed_mesh.hh>
#include <nv/gl/gl_device.hh>
#include <nv/gfx/image.hh>
#include <nv/interface/context.hh>
#include <nv/interface/window.hh>
#include <nv/interface/mesh_loader.hh>
#include <nv/io/c_file_system.hh>
#include <nv/formats/md2_loader.hh>
#include <nv/core/profiler.hh>
#include <nv/core/logging.hh>
#include <nv/core/logger.hh>
#include <nv/core/math.hh>
#include <nv/core/time.hh>
#include <nv/core/string.hh>
#include <glm/gtx/rotate_vector.hpp>

class application
{
public:
	application();
	bool initialize();
	bool run();
	~application();
protected:
	nv::device*  m_device;
	nv::window*  m_window;
	nv::context* m_context;
	nv::texture  m_diffuse;
	nv::program  m_program;

	nv::clear_state   m_clear_state;
	nv::render_state  m_render_state;
	nv::scene_state   m_scene_state;

	nv::mesh_data*       m_mesh_data;
	nv::keyframed_mesh*  m_mesh;
	nv::animation_entry* m_animation;
};

application::application()
{
	NV_PROFILE( "app_construct" );
	m_device  = new nv::gl_device();
	m_window  = m_device->create_window( 800, 600, false );
	m_context = m_window->get_context();

	nv::sampler sampler( nv::sampler::LINEAR, nv::sampler::REPEAT );
	nv::image_data* data = m_device->create_image_data( "data/manc.png" );
	m_diffuse  = m_device->create_texture( data, sampler );
	delete data;

	m_clear_state.buffers = nv::clear_state::COLOR_AND_DEPTH_BUFFER;
	m_render_state.depth_test.enabled = true;
	m_render_state.culling.enabled    = true;
	m_render_state.blending.enabled   = false;
	m_render_state.blending.src_rgb_factor   = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_rgb_factor   = nv::blending::ONE_MINUS_SRC_ALPHA;
	m_render_state.blending.src_alpha_factor = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_alpha_factor = nv::blending::ONE_MINUS_SRC_ALPHA;
}

bool application::initialize()
{
	NV_PROFILE( "app_initialize" );
	m_program = m_device->create_program( nv::slurp( "md2.vert" ), nv::slurp( "md2.frag" ) );

	nv::md2_loader* loader;
	{
		NV_PROFILE("loader->load");
		nv::c_file_system fs;
		nv::stream* mesh_file = fs.open( "data/manc.md2" );
		loader = new nv::md2_loader();
		loader->load( *mesh_file );
		m_mesh_data = loader->release_mesh_data();
		delete mesh_file;
		delete loader;
	}

	{
		NV_PROFILE("create_mesh");
		m_mesh      = new nv::keyframed_mesh_gpu( m_context, m_mesh_data, nullptr );
	}

	return true;
}

bool application::run()
{
	nv::profiler::pointer()->log_report(); 
	NV_PROFILE( "app_run" );
	int keypress = 0;

	nv::uint32 ticks   = m_device->get_ticks();
	nv::uint32 last_ticks;
	nv::fps_counter_class< nv::system_us_timer > fps_counter;

	m_animation = new nv::animation_entry( "", true, 5, 0.0f, m_mesh->get_max_frames()-1 );
	m_mesh->run_animation( m_animation );

	while(!keypress) 
	{
		last_ticks = ticks;
		ticks      = m_device->get_ticks();
		nv::uint32 elapsed = ticks - last_ticks;
		m_context->clear( m_clear_state );
		glm::vec3 eye = glm::rotate( glm::vec3( 100.0f, 25.0f, 0.0f ), (ticks / 20.f), glm::vec3( 0.0,1.0,0.0 ) );

		m_scene_state.set_model( nv::mat4(1.0f) );
		m_scene_state.get_camera().set_lookat(eye, glm::vec3(0.0f, 25.0f, 0.0f), glm::vec3(0.0, 1.0, 0.0));
		m_scene_state.get_camera().set_perspective(60.0f, 1.0f*800.0f/600.0f, 0.1f, 1000.0f);

		m_context->bind( m_diffuse, nv::TEX_DIFFUSE );
		m_device->set_opt_uniform( m_program, "light_position", glm::vec3(120.0, 120.0, 0) );
		m_device->set_opt_uniform( m_program, "light_diffuse",  glm::vec4(1.0,1.0,1.0,1.0) );
		m_device->set_opt_uniform( m_program, "light_specular", glm::vec4(1.0,1.0,1.0,1.0) );
		m_mesh->update_animation( m_animation, ticks );
		m_mesh->update( m_program );
		m_context->draw( nv::TRIANGLES, m_render_state, m_scene_state, m_program, m_mesh->get_vertex_array(), m_mesh->get_index_count() );
		m_window->swap_buffers();

		nv::io_event event;
		while(m_window->poll_event(event)) 
		{      
			switch (event.type) 
			{
			case nv::EV_QUIT:
				keypress = 1;
				break;
			case nv::EV_KEY:
				if (event.key.pressed)
				{
					switch (event.key.code) 
					{
					case nv::KEY_ESCAPE : keypress = 1; break;
					case nv::KEY_F1 : nv::profiler::pointer()->log_report(); break;
					default: break;
					}
				}
				break;
			default: break;
			}
		}
		fps_counter.tick();
	}
	return true;
}

application::~application()
{
	m_device->release( m_program );
	delete m_mesh_data;
	delete m_mesh;
	delete m_animation;
	m_device->release( m_diffuse );
	delete m_window;
	delete m_device;
}


int main(int, char* [])
{
	nv::logger log(nv::LOG_TRACE);
	log.add_sink( new nv::log_file_sink("log.txt"), nv::LOG_TRACE );
	log.add_sink( new nv::log_console_sink(), nv::LOG_TRACE );
	
	NV_LOG( nv::LOG_NOTICE, "Logging started" );
	application app;
	if ( app.initialize() )
	{
		app.run();
	}
	NV_LOG( nv::LOG_NOTICE, "Logging stopped" );

	return 0;
}

