// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file easing.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Easing interface class
 *
 * Based on http://easings.net/
 */


#ifndef NV_INTERFACE_EASING_HH
#define NV_INTERFACE_EASING_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>

namespace nv
{

	namespace detail
	{

		template < typename T >
		struct easing_back_impl
		{
			static T eval( T t )
			{
				const T s( T( 1.70158 ) );
				return t * t * ( ( s + T( 1 ) ) * t - s );
			}
		};

		template < typename T >
		struct easing_bounce_impl
		{
			static T eval( T t )
			{
				const T v = T( 1 ) - t;
				T c, d;
					 if ( v < T( 1.0 / 2.75 ) ) { c = v; d = T( 0 ); }
				else if ( v < T( 2.0 / 2.75 ) ) { c = v - T( 1.5 / 2.75 );  d = T( 0.75 ); }
				else if ( v < T( 2.5 / 2.75 ) ) { c = v - T( 2.25 / 2.75 ); d = T( 0.9375 ); }
				else { c = v - T( 2.625 / 2.75 ); d = T( 0.984375 ); }
				return T( 1 ) - ( T( 7.5625 ) * c * c + d );
			}
		};

		template < typename T >
		struct easing_circ_impl
		{
			static T eval( T t )
			{
				return T( 1 ) - nv::sqrt( T( 1 ) - t * t );
			}
		};
		
		template < typename T >
		struct easing_cubic_impl
		{
			static T eval( T t )
			{
				return t * t * t;
			}
		};

		template < typename T >
		struct easing_elastic_impl
		{
			static T eval( T t )
			{
				const T two_pi( math::two_pi<T>() );
				const T v( t - 1 );
				const T p( T( 0.3 ) );
				return -nv::pow( T(2), T(10) * v ) * nv::sin( ( v - p * T( 0.25 ) ) * two_pi / p );
			}
		};

		template < typename T >
		struct easing_expo_impl
		{
			static T eval( T t )
			{
				return t == T(0) ? T(0) : nv::pow( 2, T( 10 ) * ( t - T( 1 ) ) );
			}
		};

		template < typename T >
		struct easing_linear_impl
		{
			static T eval( T t )
			{
				return t;
			}
		};

		template < typename T >
		struct easing_quad_impl
		{
			static T eval( T t )
			{
				return t * t;
			}
		};

		template < typename T >
		struct easing_quart_impl
		{
			static T eval( T t )
			{
				const T tt = t * t;
				return tt * tt;
			}
		};

		template < typename T >
		struct easing_quint_impl
		{
			static T eval( T t )
			{
				const T tt = t * t;
				return tt * tt * t;
			}
		};

		template < typename T >
		struct easing_sine_impl
		{
			static T eval( T t )
			{
				const T half_pi( math::half_pi<T>() );
				return T( 1 ) - nv::cos( t * half_pi );
			}
		};

		template < typename T, typename Easing >
		struct easing_impl
		{
			static T eval( T t )
			{
				return Easing::eval( t );
			}

			static T in( T t )
			{
				return Easing::eval( t );
			}

			static T out( T t )
			{
				return T( 1 ) - Easing::eval( T( 1 ) - t );
			}

			static T in_out( T t )
			{
				if ( t < T( 0.5 ) )
					return Easing::eval( T( 2 ) * t ) * T( 0.5 );
				else
					return T( 1 ) - Easing::eval( T( 2 ) - ( T( 2 ) * t ) ) * T( 0.5 );
			}
		};

	}

	template < typename T, typename EaseIn, typename EaseOut >
	struct easing_composite
	{
		static T eval( T t )
		{
			if ( t < T( 0.5 ) )
				return EaseIn::eval( T( 2 ) * t ) * T( 0.5 );
			else
				return T( 1 ) - EaseOut::eval( T( 2 ) - T( 2 ) * t ) * T( 0.5 );
		}
	};

	enum class easing_type
	{
		NONE,
		BACK,
		BOUNCE,
		CIRC,
		CUBIC,
		ELASTIC,
		EXPO,
		LINEAR,
		QUAD,
		QUART,
		QUINT,
		SINE
	};

	struct easing
	{
		easing_type in;
		easing_type out;
	};

	template < typename T >
	inline T easing_eval( T t, easing_type e )
	{
		switch ( e )
		{
		case easing_type::BACK   : return detail::easing_back_impl<T>::eval( t );
		case easing_type::BOUNCE : return detail::easing_bounce_impl<T>::eval( t );
		case easing_type::CIRC   : return detail::easing_circ_impl<T>::eval( t );
		case easing_type::CUBIC  : return detail::easing_cubic_impl<T>::eval( t );
		case easing_type::ELASTIC: return detail::easing_elastic_impl<T>::eval( t );
		case easing_type::EXPO   : return detail::easing_expo_impl<T>::eval( t );
		case easing_type::LINEAR : return detail::easing_linear_impl<T>::eval( t );
		case easing_type::QUAD   : return detail::easing_quad_impl<T>::eval( t );
		case easing_type::QUART  : return detail::easing_quart_impl<T>::eval( t );
		case easing_type::QUINT  : return detail::easing_quint_impl<T>::eval( t );
		case easing_type::SINE   : return detail::easing_sine_impl<T>::eval( t );
		default:
			break;
		}
		return T( 0 );
	};

	template < typename T >
	inline T easing_eval( T t, easing_type in, easing_type out )
	{
		if ( in != easing_type::NONE )
		{
			if ( out != easing_type::NONE )
			{
				if ( t < T( 0.5 ) )
					return easing_eval( T( 2 ) * t, in ) * T( 0.5 );
				else
					return T( 1 ) - easing_eval( T( 2 ) - T( 2 ) * t, out ) * T( 0.5 );
			}
			else
				return easing_eval( t, in );
		}
		else if ( out != easing_type::NONE )
		{
			return T( 1 ) - easing_eval( T( 1 ) - t, out );
		}
		return T( 0 );
	}

	template < typename T >
	inline T easing_eval( T t, easing e )
	{
		return easing_eval( t, e.in, e.out );
	}

}

#endif // NV_INTERFACE_EASING_HH
