// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_GFX_TEXTURE_FONT_HH
#define NV_GFX_TEXTURE_FONT_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/unordered_map.hh>
#include <nv/stl/math.hh>
#include <nv/gfx/texture_atlas.hh>

namespace nv
{
	struct texture_glyph
	{
		uint16 charcode;//!< Character code value.
		ivec2  size;    //!< Width and height of the glyph.
		ivec2  offset;  //!< Offset of the glyph's position from the base line.
		vec2   advance; //!< Cursor advance distance.
		vec2   tl;      //!< Top-left of the glyph's bounding box.
		vec2   br;      //!< Bottom-right of the glyph's bounding box.
		unordered_map< uint16, float > kerning; //!< Kerning space between other characters.

		/**
		 * Creates a new texture_glyph.
		 */
		texture_glyph();

		/**
		 * Gets the kerning space between this character and the requested character.
		 *
		 * @param other The character to get the spacing for.
		 * @returns The amount of space for kerning.
		 */
		float get_kerning( const uint16 other ); 
	};

	class texture_font
	{
		public:
			texture_font( texture_atlas* atlas, const string_view& filename, float size );
			const texture_glyph* get_glyph( uint16 charcode ) const;
			bool load_glyphs( string_view codes );
			float get_size() const { return m_size; }
			~texture_font();
		private:
			void generate_kerning();
		private:
			unordered_map< uint16, texture_glyph > m_glyphs; //!< Hash table of glyphs for this font.

			texture_atlas* m_atlas; //!< Atlas Image object for this font.
			string128 m_filename;//!< Filename
			float m_size;           //!< Font size.
			float m_height;         //!< Height of the font. (x-height?)
			float m_linegap;        //!< Amount of space between lines.
			float m_ascender;       //!< Height of ascender lines (lines extending above the glyph's x-height).
			float m_descender;      //!< Height of descender lines (lines extending below the glyph's x-height).
			bool m_hinting;         //!< Whether or not glyph hints are used.
			bool m_filtering;       //!< Whether or not glyphs are color filtered for LCD displays.
			uint8 m_lcd_weights[5]; //!< Color filtering weights.
			void* m_rlibrary;       //!< Pointer to the library.
			void* m_rface;          //!< Pointer to the font face.
	};
}

#endif // NV_GFX_TEXTURE_FONT_HH

