// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file terminal.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief resource interface
*/

#ifndef NV_CORE_RESOURCE_HH
#define NV_CORE_RESOURCE_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/hash_store.hh>
#include <nv/stl/vector.hh>

#include <nv/core/logging.hh>


namespace nv
{

	struct resource_unchecked {};
	template < typename T > class resource;
	template < typename T > class resource_handle;
	template < typename T > class resource_lock;
	using resource_id = shash64;
	using resource_type_id = shash64;

	class resource_handler
	{
	public:
		resource_handler( resource_type_id id ) : m_type_id( id ) {}

	protected:
		virtual const void* raw_lock( resource_id id, bool checked = true ) = 0;
		virtual void raw_add( resource_id id, void* value ) = 0;
		virtual void unlock( resource_id id ) = 0;
		virtual void release( resource_id id ) = 0;
		virtual bool exists( resource_id id ) = 0;
		virtual bool load_resource( const string_view& id ) = 0;

		template< typename T >
		const T* lock( resource_id id, bool checked = true )
		{
			NV_ASSERT( rtti_type_hash<T>::hash() == m_type_id.value(), "Wrong type lock!" );
			return reinterpret_cast< const T* >( raw_lock( id, checked ) );
		}
		template< typename T >
		resource< T > create( resource_id id )
		{
			NV_ASSERT( rtti_type_hash<T>::hash() == m_type_id.value(), "Wrong type creation!" );
			resource< T > result;
			result.m_id = id;
			result.m_handler = this;
			return result;
		}



		template < typename T >
		friend class resource;
		template < typename T >
		friend class resource_handle;
		template < typename T >
		friend class resource_lock;
		friend class resource_manager;

		resource_type_id m_type_id;
	};


	template < typename T >
	class resource 
	{
	public:
		resource() : m_id(), m_handler( nullptr ) {}
		resource_id id() const { return m_id; }
		constexpr bool is_valid() const { return m_id && m_handler; }
		constexpr explicit operator bool() const { return is_valid(); }
		resource_lock< T > lock() const { return resource_lock< T >( *this, resource_unchecked() ); }
		~resource()
		{
			if ( m_handler ) m_handler->release( m_id );
		}
	protected:
		resource_id       m_id;
		resource_handler* m_handler;

		template < typename U >
		friend constexpr bool operator==( const resource< U >&, const resource< U >& );
		template < typename U >
		friend constexpr bool operator!=( const resource< U >&, const resource< U >& );
		template < typename U >
		friend constexpr bool operator<( const resource< U >&, const resource< U >& );
		template < typename U >
		friend constexpr bool operator>( const resource< U >&, const resource< U >& );

		friend class resource_lock< T >;
		friend class resource_handler;
	};

	template < typename T >
	constexpr bool operator== ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id == rhs.m_id && lhs.m_handler == rhs.m_handler;
	}

	template < typename T >
	constexpr bool operator!= ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id != rhs.m_id || lhs.m_handler != rhs.m_handler;
	}

	template < typename T >
	constexpr bool operator< ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id.value() < rhs.m_id.value();
	}

	template < typename T >
	constexpr bool operator> ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id.value() > rhs.m_id.value();
	}

	template < typename T >
	class resource_handle
	{
	public:

		void release( resource_handler* r )
		{
			r->release( m_id );
			m_id = 0;
		}
		~resource_handle()
		{
			NV_ASSERT( m_id == 0, "Resource not released!" );
		}
	protected:
		resource_id       m_id;

		friend class resource_lock< T >;
		friend class resource_handler;
	};

	template < typename T >
	class resource_lock
	{
	public:
		resource_lock() = delete;
		explicit resource_lock( const resource< T >& r, resource_unchecked )
			: m_id( r.m_id )
			, m_handler( r.m_handler )
			, m_resource( r.m_handler ? r.m_handler->lock<T>( r.m_id, false ) : nullptr ) {}
		explicit resource_lock( const resource< T >& r ) 
			: m_id( r.m_id )
			, m_handler( r.m_handler )
			, m_resource( r.m_handler->lock<T>( r.m_id ) ) {}
		explicit resource_lock( const resource_handle< T >& r, resource_handler* handler ) : m_id( r.m_id ), m_handler( handler ), m_resource( handler->raw_lock( r.m_id ) ) {}
		resource_lock( const resource_lock& ) = delete;
		resource_lock& operator=( const resource_lock& other ) = delete;
		resource_lock( resource_lock&& other )
		{
			m_id       = other.m_id;
			m_handler  = other.m_handler;
			m_resource = other.m_resource;
			other.m_handler = nullptr;
		}
		resource_lock& operator=( resource_lock&& other )
		{
			if ( this != &other )
			{
				if ( m_handler ) m_handler->unlock( m_id );
				m_id       = other.m_id;
				m_handler  = other.m_handler;
				m_resource = other.m_resource;
				other.m_handler = nullptr;
			}
			return *this;
		}
		const T& operator*() const { return *m_resource; }
		const T* operator->() const { return m_resource; }
		constexpr bool is_valid() const { return m_resource != nullptr; }
		constexpr explicit operator bool() const { return is_valid(); }
		~resource_lock()
		{
			if ( m_handler ) m_handler->unlock( m_id );
		}
	private:
		resource_id       m_id;
		resource_handler* m_handler;
		const T*          m_resource;
	};


	class resource_manager
	{
	public:
		template < typename T, typename Manager >
		Manager* register_resource_handler( Manager* handler )
		{
			return static_cast<Manager*>( register_resource_handler( handler, resource_type_id( rtti_type_hash<T>::value ) ) );
		}

		template < typename T >
		resource< T > get( const string_view& id )
		{
			auto m = m_handlers.find( resource_type_id( rtti_type_hash<T>::value ) );
			NV_ASSERT( m != m_handlers.end(), "Resource type unrecognized!" );
			if ( m->second->exists( id ) || m->second->load_resource( id ) )
				return m->second->create< T >( id );
			NV_ASSERT( false, "resource_manager.get failed!" );
			return resource< T >();
		}

		template < typename T >
		resource< T > get( uint64 id )
		{
			auto m = m_handlers.find( resource_type_id( rtti_type_hash<T>::value ) );
			NV_ASSERT( m != m_handlers.end(), "Resource type unrecognized!" );
			if ( m->second->exists( id ) )
			{
				return m->second->create< T >( id );
			}
			NV_ASSERT( false, "resource_manager.get failed!" );
			return resource< T >();
		}

		template < typename T >
		resource< T > add( shash64 id, T* value )
		{
			auto m = m_handlers.find( resource_type_id( rtti_type_hash<T>::value ) );
			NV_ASSERT( m != m_handlers.end(), "Resource type unrecognized!" );
			m->second->raw_add( id, value );
			return m->second->create< T >( id );
		}

		virtual ~resource_manager()
		{
			// reverse destruction order
			for ( auto i = m_handler_list.rbegin(); i != m_handler_list.rend(); ++i )
				delete *i;
		}
	protected:

		resource_handler* register_resource_handler( resource_handler* handler, resource_type_id type_hash )
		{
			NV_ASSERT( m_handlers.find( type_hash ) == m_handlers.end(), "Handler already registered!" );
			m_handlers[type_hash] = handler;
			m_handler_list.push_back( handler );
			return handler;
		}

	protected:
		// needed for destruction order
		vector< resource_handler* > m_handler_list;
		hash_store< resource_type_id, resource_handler* > m_handlers;
	};
}

#endif // NV_CORE_RESOURCE_HH

