// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file profiler.hh
 * @author Kornel Kisielewicz
 * @brief profiler
 */

#ifndef NV_CORE_PROFILER_HH
#define NV_CORE_PROFILER_HH

#include <nv/common.hh>
#include <nv/stl/singleton.hh>
#include <nv/stl/string.hh>
#include <nv/stl/string_map.hh>

#if NV_PROFILER 
#define NV_PROFILE( tag ) nv::profiler_guard __profile( tag##_hls64 )
#define NV_PROFILE_IF( tag, condition ) nv::profiler_condition_guard __profile( tag##_hls64, condition )
#else
#define NV_PROFILE( tag )
#define NV_PROFILE_IF( tag, condition ) 
#endif


namespace nv
{
	class profiler : public auto_singleton< profiler >
	{
	protected:
		typedef hashed_literal_string_64 string_type;

		class node
		{
			friend class profiler;
		public:
			~node();
		protected:
			node( string_view tag, node* parent );
			node* get_parent() { return m_parent; }
			node* request_child( string_type&& tag );
			void start();
			bool stop();

			typedef literal_map< node* > map;

			string_view m_tag;
			map         m_children;
			node*       m_parent;
			uint32      m_recusion;

			uint32      m_calls;
			uint64      m_start_time_us;
			uint64      m_total_time_us;
		};

	protected:
		profiler();
		~profiler();

		void start_profile( string_type&& tag );
		void stop_profile();
	public:
		friend class auto_singleton< profiler >;
		friend class profiler_guard;
		friend class profiler_condition_guard;
		void log_report();
	private:
		void log_node_children( uint32 indent, const node* n );
		node* m_root;
		node* m_current;
	};

	class profiler_guard
	{
	public:
		profiler_guard( hashed_literal_string_64&& tag )
		{
			profiler::pointer()->start_profile( ::nv::move( tag ) );
		}

		~profiler_guard()
		{
			profiler::pointer()->stop_profile();
		}
	};

	class profiler_condition_guard
	{
	public:
		profiler_condition_guard( hashed_literal_string_64&& tag, bool condition )
			: m_active( condition )
		{
			if ( m_active ) profiler::pointer()->start_profile( ::nv::move( tag ) );
		}

		~profiler_condition_guard()
		{
			if ( m_active ) profiler::pointer()->stop_profile();
		}
	private:
		bool m_active;
	};

} // namespace nv

#endif // NV_CORE_PROFILER_HH
