// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file common.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief type traits - common traits used everywhere
 */

#ifndef NV_STL_TRAITS_COMMON_HH
#define NV_STL_TRAITS_COMMON_HH

#include <nv/common.hh>

namespace nv
{

	template< typename T, T VALUE>
	struct integral_constant
	{
		static constexpr T value = VALUE;
		typedef T value_type;
		typedef integral_constant<T, VALUE> type;
		constexpr operator value_type() const noexcept { return value; }
		constexpr value_type operator()() const noexcept { return value; }
	};

	// TODO: Propagate
	template< bool B >
	using bool_constant = integral_constant < bool, B > ;

	typedef bool_constant<true> true_type;
	typedef bool_constant<false> false_type;

	template< bool Test, typename T = void>
	struct enable_if {};

	template< typename T >
	struct enable_if < true, T >
	{
		typedef T type;
	};

	template< bool Test, typename T = void>
	using enable_if_t = typename enable_if< Test, T >::type;

	template< bool Test, typename T1, typename T2 >
	struct conditional
	{
		typedef T2 type;
	};

	template< typename T1, typename T2>
	struct conditional < true, T1, T2 >
	{
		typedef T1 type;
	};

	template< bool Test, typename T1, typename T2 >
	using conditional_t = typename conditional< Test, T1, T2 >::type;

	template < typename T1, typename T2 >
	struct is_same : false_type {};

	template < typename T >
	struct is_same < T, T > : true_type{};

	// TODO: these seem to simple compared to MSVC/GCC - so we'll leave them in
	// detail - research why it is so.
	template < typename T >	struct is_const : false_type {};
	template < typename T > struct is_const < T const > : true_type{};
	template < typename T > struct is_volatile : false_type {};
	template < typename T > struct is_volatile < T volatile > : true_type{};
	// TODO END

	template < typename T >	struct is_lvalue_reference : false_type {};
	template < typename T >	struct is_lvalue_reference < T& > : true_type{};
	template < typename T >	struct is_rvalue_reference : false_type {};
	template < typename T >	struct is_rvalue_reference < T&& > : true_type{};

	template < typename T >
	struct is_reference : bool_constant < 
		is_lvalue_reference<T>::value || 
		is_rvalue_reference<T>::value 
	> {};

	// add const volatile and reference mutators

	template< typename T > struct add_const          { typedef const T type; };
	template< typename T > struct add_volatile       { typedef volatile T type; };
	template< typename T > struct add_cv             { typedef const volatile T type; };
	
	template< typename T > struct add_reference             
	{
		typedef conditional< is_reference<T>::value, T, T&& > type;
	};
	template<> struct add_reference < void >                { typedef void type; };
	template<> struct add_reference < const void >          { typedef void type; };
	template<> struct add_reference < volatile void >       { typedef void type; };
	template<> struct add_reference < const volatile void > { typedef void type; };

	template< typename T > struct add_rvalue_reference             { typedef T&& type; };
	template< typename T > struct add_rvalue_reference < T& >      { typedef T   type; };
	template<> struct add_rvalue_reference < void >                { typedef void type; };
	template<> struct add_rvalue_reference < const void >          { typedef void type; };
	template<> struct add_rvalue_reference < volatile void >       { typedef void type; };
	template<> struct add_rvalue_reference < const volatile void > { typedef void type; };

	template< typename T > using add_lvalue_reference = add_reference<T>;

	template< typename T > using add_const_t            = typename add_const<T>::type;
	template< typename T > using add_volatile_t         = typename add_volatile<T>::type;
	template< typename T > using add_cv_t               = typename add_cv<T>::type;
	template< typename T > using add_reference_t        = typename add_reference<T>::type;
	template< typename T > using add_lvalue_reference_t = typename add_lvalue_reference<T>::type;
	template< typename T > using add_rvalue_reference_t = typename add_rvalue_reference<T>::type;

	// remove const volatile and reference mutators

	template< typename T > struct remove_reference { typedef T type; };
	template< typename T > struct remove_reference < T& >                     { typedef T type; };
	template< typename T > struct remove_reference < T&& >                    { typedef T type; };

	template< typename T > struct remove_const                                { typedef T type; };
	template< typename T > struct remove_const < const T >                    { typedef T type; };
	template< typename T > struct remove_const < const T[] >                  { typedef T type[]; };
	template< typename T, unsigned int N > struct remove_const < const T[N] > { typedef T type[N]; };

	template< typename T > struct remove_volatile                             { typedef T type; };
	template< typename T > struct remove_volatile < volatile T >              { typedef T type; };
	template< typename T > struct remove_volatile < volatile T[] >            { typedef T type[]; };
	template< typename T, unsigned int N > struct remove_volatile < volatile T[N] > { typedef T type[N]; };

	template< typename T >
	struct remove_cv
	{
		typedef typename remove_const< typename remove_volatile<T>::type >::type type;
	};

	template< typename T >
	struct remove_cvr
	{
		typedef typename remove_cv< typename remove_reference<T>::type >::type type;
	};

	template< typename T > using remove_reference_t = typename remove_reference<T>::type;
	template< typename T > using remove_const_t     = typename remove_const<T>::type;
	template< typename T > using remove_volatile_t  = typename remove_volatile<T>::type;
	template< typename T > using remove_cvr_t       = typename remove_cvr<T>::type;
	template< typename T > using remove_cv_t        = typename remove_cv<T>::type;

	// pointer/array mutators

	template< typename T > struct add_pointer { typedef typename remove_reference<T>::type *type; };
	
	template< typename T > struct remove_pointer { typedef T type; };
	template< typename T > struct remove_pointer < T* > { typedef T type; };
	template< typename T > struct remove_pointer < T* const > { typedef T type; };
	template< typename T > struct remove_pointer < T* volatile > { typedef T type; };
	template< typename T > struct remove_pointer < T* const volatile > { typedef T type; };

	template< typename T > struct remove_extent { typedef T type; };
	template< typename T > struct remove_extent < T[] > { typedef T type; };
	template< typename T, unsigned int N > struct remove_extent < T[N] > { typedef T type; };

	template< typename T > struct remove_all_extents { typedef T type; };
	template< typename T > struct remove_all_extents < T[] > { typedef typename remove_all_extents<T>::type type; };
	template< typename T, unsigned int N > struct remove_all_extents < T[N] > { typedef typename remove_all_extents<T>::type type; };

	template< typename T > using add_pointer_t        = typename add_pointer<T>::type;
	template< typename T > using remove_pointer_t     = typename remove_pointer<T>::type;
	template< typename T > using remove_extent_t      = typename remove_extent<T>::type;
	template< typename T > using remove_all_extents_t = typename remove_all_extents<T>::type;

	template < typename T >
	typename add_rvalue_reference<T>::type declval();

	template< typename T >
	constexpr typename remove_reference<T>::type&& move( T&& arg ) noexcept
	{
		return static_cast<typename remove_reference<T>::type&&>( arg );
	}

	template < typename T >
	constexpr T&& forward( typename remove_reference<T>::type& t ) noexcept
	{
		return static_cast<T&&>( t );
	}

	template < typename T >
	constexpr T&& forward( typename remove_reference<T>::type&& t ) noexcept
	{
		static_assert( !is_lvalue_reference<T>::value, "Can not forward an rvalue as an lvalue." );
		return static_cast<T&&>( t );
	}

	namespace detail
	{
		template< typename T >
		struct addressof_helper
		{
			T & value;
			constexpr addressof_helper( T & v ) : value( v ) {}
			constexpr operator T& ( ) const { return value; }
		private:
			addressof_helper & operator=( const addressof_helper & );
		};

		template< typename T >
		struct addressof_impl
		{
			static constexpr T * f( T & v, long )
			{
				return reinterpret_cast<T*>(
					&const_cast<char&>( reinterpret_cast<const volatile char &>( v ) ) );
			}
			static constexpr T * f( T * v, int ) { return v; }
		};
	}

	template< typename T >
	T * addressof( T & v )
	{
		return detail::addressof_impl<T>::f( detail::addressof_helper<T>( v ), 0 );
	}


	template < typename T >
	struct extract_value_type
	{
		typedef typename T::value_type type;
	};

	template < typename T >
	struct extract_size_type
	{
		typedef typename T::size_type type;
	};

	template < typename T >
	using value_type_t = typename extract_value_type<T>::type;
	template < typename T >
	using size_type_t = typename extract_size_type<T>::type;

	namespace detail
	{
		struct sfinae_types
		{
			typedef char size_one;
			typedef struct { char unused[2]; } size_two;
		};
	}
}

// see unsafe (experimental) version in type_traits experimental.hh
#define NV_GENERATE_HAS_TYPE_SAFE( TYPE ) \
namespace macro_detail { \
template < typename T > class has_##TYPE##_impl : nv::detail::sfinae_types {\
  template< typename U > struct wrap_type {}; \
  template< typename U > static size_one test( wrap_type< typename U::TYPE >* ); \
  template< typename U > static size_two test( ... ); \
public:\
  static constexpr bool value = sizeof( test<T>(0) ) == 1; \
}; } \
template < typename T > \
struct has_##TYPE : nv::bool_constant< macro_detail::has_##TYPE##_impl< remove_cv_t<T> >::value > {};

#endif // NV_STL_TRAITS_COMMON_HH
