// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_INTERFACE_MESH_DATA_HH
#define NV_INTERFACE_MESH_DATA_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/string.hh>
#include <nv/interface/data_channel.hh>
#include <nv/gfx/animation.hh>

namespace nv
{

	struct index_u16 { uint16 index; };
	struct index_u32 { uint32 index; };

	//using mesh_data = data_channel_set;
	// TODO : attribute/property implementation and read/write on every nmd::command

	class child_list : noncopyable
	{
		static constexpr size_t MAX_CHILDREN = 7;
	public:
		child_list() : m_size( 0 )
		{
			nv::raw_fill_n( m_data, MAX_CHILDREN, 0 );
		}
		void push( uint8 id )
		{
			NV_ASSERT( m_size < MAX_CHILDREN, "Range error" );
			m_data[m_size++] = id;
		}
		const uint8* begin() const { return m_data; }
		const uint8* end() const { return m_data + m_size; }
	private:
		uint8 m_size;
		uint8 m_data[MAX_CHILDREN];
	};
	static_assert( sizeof( child_list ) == 8, "Align/Padding fail for child_list" );

	class data_node_list
	{
		friend class mesh_creator;
		friend class mesh_nodes_creator;
		friend class data_node_list_creator;
	public:
		typedef vector< data_node_info > storage;
		typedef storage::const_iterator  const_iterator;

		explicit data_node_list( shash64 name )
			: m_name( name )
		{
		}

		void assign( const data_node_list& other )
		{
			m_name = other.m_name;
			for ( auto node : other )
				m_data.assign( other.m_data );
		}

		data_node_list( data_node_list&& other )
			: m_name( other.m_name ), m_data( nv::move( other.m_data ) )
		{
		}

		void append( const data_node_info& set )
		{
			m_data.push_back( set );
		}

		size_t size() const { return m_data.size(); }
		bool empty() const { return m_data.empty(); }

		const data_node_info& operator[]( size_t i ) const
		{
			return m_data[i];
		}

		const_iterator begin() const { return m_data.begin(); }
		const_iterator end() const { return m_data.end(); }

		shash64 get_name() const { return m_name; }

	protected:
		vector< data_node_info > m_data;
		shash64                  m_name;
	};


	class data_node_tree : public data_node_list
	{
		friend class mesh_creator;
		friend class mesh_nodes_creator;
		friend class data_node_list_creator;
	public:
		
		explicit data_node_tree( shash64 name ) : data_node_list( name ) {}

		void initialize()
		{
			m_children.resize( size() );
			for ( uint8 n = 0; n < uint8( m_data.size() ); ++n )
			{
				if ( m_data[n].parent_id != -1 )
					m_children[m_data[n].parent_id].push( n );
			}
		}

		const child_list& children( size_t i ) const
		{
			return m_children[i];
		}

	protected:
		vector< child_list >     m_children;
	};


	class mesh_nodes_data
	{
		friend class mesh_nodes_creator;
	public:
		typedef vector< data_channel_set* > storage;
		typedef storage::const_iterator const_iterator;

		explicit mesh_nodes_data( shash64 name )
			: m_info( name ), m_frame_rate(0), m_frame_count(0)
		{
		}

		explicit mesh_nodes_data( shash64 name, uint16 a_fps, uint16 a_frames )
			: m_info( name ), m_frame_rate(a_fps), m_frame_count(a_frames)
		{
		}

		void append( data_channel_set* set, data_node_info info )
		{
			m_info.append( info );
			m_data.push_back( set );
		}

		void initialize()
		{
			m_info.initialize();
		}

		const child_list& children( size_t i ) const
		{
			return m_info.children(i);
		}

		size_t size() const { return m_data.size(); }
		bool empty() const { return m_data.empty(); }

		bool is_animated( size_t i ) const
		{
			if ( i >= m_data.size() ) return false;
			return ( m_data[i]->size() > 0 );
		}

		bool is_animated() const
		{
			for ( auto data : m_data )
			{
				if ( data->size() > 0 ) return true;
			}
			return false;
		}

		const data_channel_set* get_by_hash( shash64 h ) const
		{
			for ( uint32 i = 0; i < m_info.size(); ++i )
			{
				if ( m_info[i].name == h )
					return m_data[i];
			}
			return nullptr;
		}

		int get_index_by_hash( shash64 h ) const
		{
			for ( uint32 i = 0; i < m_data.size(); ++i )
			{
				if ( m_info[ i ].name == h )
					return int( i );
			}
			return -1;
		}

		const data_channel_set* operator[]( size_t i ) const
		{
			return m_data[i];
		}
		const data_node_info& get_info( size_t i ) const
		{
			return m_info[i];
		}
		const data_node_tree& get_info() const
		{
			return m_info;
		}

		const_iterator begin() const { return m_data.begin(); }
		const_iterator end() const { return m_data.end(); }

		uint16 get_fps() const { return m_frame_rate; }
		uint16 get_frame_count() const { return m_frame_count; }
		shash64 get_name() const { return m_info.get_name(); }

		~mesh_nodes_data()
		{
			for ( auto data : m_data ) delete data;
		}

	private:
		data_node_tree              m_info;
		vector< data_channel_set* > m_data;
		uint16  m_frame_rate;
		uint16  m_frame_count;
	};

}

#endif // NV_INTERFACE_MESH_DATA_HH
