// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file animation.hh
 * @author Kornel Kisielewicz
 * @brief animation
 */

#ifndef NV_ENGINE_ANIMATION_HH
#define NV_ENGINE_ANIMATION_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/gfx/skeleton_instance.hh>

namespace nv
{
	class animator_bind_data;
	class animator_data;
}

NV_RTTI_DECLARE_NAME( nv::mesh_nodes_data, "nv::mesh_nodes_data" )
NV_RTTI_DECLARE_NAME( nv::animator_bind_data, "nv::animator_bind_data" )
NV_RTTI_DECLARE_NAME( nv::animator_data, "nv::animator_data" )

namespace nv
{

	class animator_clip_data
	{
	public:
		animator_clip_data( resource< mesh_nodes_data > data, frame_range frange )
			: m_data( data )
			, m_range( frange )
		{
		}
		// 	uint32 get_start_frame() const { return m_range.start; }
		// 	uint32 get_end_frame() const { return m_range.end; }
		// 	uint32 get_frame_count() const { return m_range.duration(); }
		// 	bool is_looping() const { return m_range.is_looping; }
		resource< mesh_nodes_data > get_data() const { return m_data; }
		const frame_range& get_range() const { return m_range; }
	protected:
		resource< mesh_nodes_data > m_data;
		frame_range                 m_range;
	};

	class animator_data
	{
	public:
		animator_data() {}
		const animator_clip_data* get_clip( shash64 name ) const
		{
			auto it = m_clip_data.find( name );
			return ( it != m_clip_data.end() ? &it->second : nullptr );
		}
		bool has_clip( shash64 name ) const
		{
			auto it = m_clip_data.find( name );
			return it != m_clip_data.end();
		}
		void add_clip( shash64 name, animator_clip_data&& entry )
		{
			m_clip_data.assign( name, entry );
		}
	protected:
		hash_store< shash64, animator_clip_data > m_clip_data;
	};

	// resource per animator/model set
	class animator_bind_data
	{
	public:
		animator_bind_data( const data_node_list& bones )
			: m_bone_list( bones.get_name() )
		{
			// TODO: m_bone_list.clone
			for ( auto bone : bones )
			{
				m_bone_list.append( bone );
			}
			m_bone_transforms.prepare( m_bone_list );
		}
		void add_binding( shash64 id, const mesh_nodes_data* nodes )
		{
			m_bind_data[id].prepare( nodes, m_bone_list );
		}

		bool has_binding( shash64 id ) const { return m_bind_data.find( id ) != m_bind_data.end(); }
		const bone_transforms& get_bone_transforms() const { return m_bone_transforms; }
		const skeleton_binding& get_binding( shash64 id ) const
		{
			return m_bind_data.at( id );
		}
	protected:
		hash_store< shash64, skeleton_binding > m_bind_data;
		bone_transforms                         m_bone_transforms;
		data_node_list                          m_bone_list;
	};

	// per model instance
	class animator_instance
	{
	public:
		animator_instance() {}
		void initialize( resource< animator_data > data, resource< animator_bind_data > bind_data )
		{
			m_data = data;
			m_bind_data = bind_data;
		}
		bool is_valid() const { return m_data.is_valid() && m_bind_data.is_valid(); }
		bool has_clip( shash64 id ) const
		{
			if ( auto data = m_data.lock() )
				return data->has_clip( id );
			else
				return false;
		}
		mat4 get_transform( uint32 idx ) const
		{
			if ( idx < m_skeleton.size() ) return m_skeleton.transforms()[idx];
			return mat4();
		}
		const skeleton_instance& get_skeleton() const
		{
			return m_skeleton;
		}

		void update( shash64 clip, uint32 time )
		{
			if ( is_valid() )
			{
				auto data = m_data.lock();
				auto bind_data = m_bind_data.lock();
				const animator_clip_data* clip_data = data->get_clip( clip );
				if ( clip_data )
				{
					if ( auto nodes = clip_data->get_data().lock() )
					{
						float fframe = clip_data->get_range().frame_from_ms_fps( time, nodes->get_fps() );
						m_transforms.animate( &*nodes, bind_data->get_binding( clip ), fframe );
					}
				}
				m_skeleton.assign( m_transforms, bind_data->get_bone_transforms() );
			}
		}
	protected:
		resource< animator_data >      m_data;
		resource< animator_bind_data > m_bind_data;
		skeleton_transforms            m_transforms;
		skeleton_instance              m_skeleton;
	};


}

#endif // NV_ENGINE_ANIMATION_HH
