// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file basic.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief basic matrix ops
 */

#ifndef NV_STL_MATH_MATRIX_HH
#define NV_STL_MATH_MATRIX_HH

#include <nv/stl/math/common.hh>
#include <nv/stl/math/relational.hh>
#include <nv/stl/math/mat2.hh>
#include <nv/stl/math/mat3.hh>
#include <nv/stl/math/mat4.hh>

namespace nv
{

	namespace math
	{

		namespace detail
		{
			template < typename T >
			struct vec_to_mat;

			template < typename T >
			struct vec_to_mat< tvec2< T > >
			{
				typedef tmat2< T > type;
			};

			template < typename T >
			struct vec_to_mat< tvec3< T > >
			{
				typedef tmat3< T > type;
			};

			template < typename T >
			struct vec_to_mat< tvec4< T > >
			{
				typedef tmat4< T > type;
			};
		}

		template < typename T, typename enable_if< is_mat<T>::value >::type* = nullptr >
		inline T matrix_comp_mult( const T& x, const T& y )
		{
			T result( no_init );
			for ( size_t i = 0; i < nv::size( result ); ++i )
				result[i] = x[i] * y[i];
			return result;
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline typename detail::vec_to_mat<T>::type outer_product( const T& c, const T& r )
		{
			typename detail::vec_to_mat<T>::type result( no_init );
			for ( size_t i = 0; i < nv::size( result ); ++i )
				result[i] = c * r[i];
			return result;
		}

		template < typename T >
		inline tmat2<T> transpose( const tmat2<T>& m )
		{
			tmat2<T> result( no_init );
			result[0][0] = m[0][0]; result[0][1] = m[1][0];
			result[1][0] = m[0][1]; result[1][1] = m[1][1];
			return result;
		}

		template < typename T >
		inline tmat3<T> transpose( const tmat3<T>& m )
		{
			tmat3<T> result( no_init );
			result[0][0] = m[0][0]; result[0][1] = m[1][0]; result[0][2] = m[2][0];
			result[1][0] = m[0][1]; result[1][1] = m[1][1]; result[1][2] = m[2][1];
			result[2][0] = m[0][2]; result[2][1] = m[1][2]; result[2][2] = m[2][2];
			return result;
		}

		template < typename T >
		inline tmat4<T> transpose( const tmat4<T>& m )
		{
			tmat4<T> result( no_init );
			result[0][0] = m[0][0]; result[0][1] = m[1][0]; result[0][2] = m[2][0]; result[0][3] = m[3][0];
			result[1][0] = m[0][1]; result[1][1] = m[1][1]; result[1][2] = m[2][1]; result[1][3] = m[3][1];
			result[2][0] = m[0][2]; result[2][1] = m[1][2]; result[2][2] = m[2][2]; result[2][3] = m[3][2];
			result[3][0] = m[0][3]; result[3][1] = m[1][3]; result[3][2] = m[2][3]; result[3][3] = m[3][3];
			return result;
		}

		template < typename T >
		inline T determinant( const tmat2<T>& m )
		{
			return m[0][0] * m[1][1] - m[1][0] * m[0][1];
		}

		template < typename T >
		inline T determinant( const tmat3<T>& m )
		{
			return  + m[0][0] * ( m[1][1] * m[2][2] - m[2][1] * m[1][2] )
					- m[1][0] * ( m[0][1] * m[2][2] - m[2][1] * m[0][2] )
					+ m[2][0] * ( m[0][1] * m[1][2] - m[1][1] * m[0][2] );
		}

		template < typename T >
		inline T determinant( const tmat4<T>& m )
		{
			T sub0 = m[2][2] * m[3][3] - m[3][2] * m[2][3];
			T sub1 = m[2][1] * m[3][3] - m[3][1] * m[2][3];
			T sub2 = m[2][1] * m[3][2] - m[3][1] * m[2][2];
			T sub3 = m[2][0] * m[3][3] - m[3][0] * m[2][3];
			T sub4 = m[2][0] * m[3][2] - m[3][0] * m[2][2];
			T sub5 = m[2][0] * m[3][1] - m[3][0] * m[2][1];

			tvec4<T> cf(
				+( m[1][1] * sub0 - m[1][2] * sub1 + m[1][3] * sub2 ),
				-( m[1][0] * sub0 - m[1][2] * sub3 + m[1][3] * sub4 ),
				+( m[1][0] * sub1 - m[1][1] * sub3 + m[1][3] * sub5 ),
				-( m[1][0] * sub2 - m[1][1] * sub4 + m[1][2] * sub5 )
			);

			return m[0][0] * cf[0] + m[0][1] * cf[1] + m[0][2] * cf[2] + m[0][3] * cf[3];
		}

		template < typename T, typename enable_if< is_fp_mat< T >::value >::type* = nullptr >
		inline T inverse( const T& m )
		{
			return detail::compute_inverse( m );
		}

	}

}

#endif // NV_STL_MATH_MATRIX_HH
