// Copyright (C) 2014-2015 ChaosForge Ltd 
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_LIB_ASSIMP_HH
#define NV_LIB_ASSIMP_HH

#include <nv/common.hh>
#include <nv/core/logging.hh>
#include <nv/stl/string.hh>
#include <nv/stl/math.hh>

#define NV_ASSIMP_DYNAMIC
//#define NV_ASSIMP_SHARED

#if NV_PLATFORM == NV_WINDOWS
#	define NV_ASSIMP_PATH "assimp32.dll"
#elif NV_PLATFORM == NV_APPLE
#	define NV_ASSIMP_PATH ""
#else
#	define NV_ASSIMP_PATH "libassimp32.so"
#endif

extern "C" {

#if NV_PLATFORM == NV_WINDOWS 
#	define NV_ASSIMP_APIENTRY 
#else
#	define NV_ASSIMP_APIENTRY
#endif

#if defined(NV_ASSIMP_SHARED) && (NV_PLATFORM == NV_WINDOWS)
#	define NV_ASSIMP_API __declspec(dllimport)
#else
#	define NV_ASSIMP_API extern
#endif

#include <nv/lib/detail/assimp_types.inc>

#if defined(NV_ASSIMP_DYNAMIC)
#	define NV_ASSIMP_FUN( rtype, fname, fparams ) NV_ASSIMP_API rtype ( NV_ASSIMP_APIENTRY *fname) fparams
#else
#	define NV_ASSIMP_FUN( rtype, fname, fparams ) NV_ASSIMP_API rtype NV_ASSIMP_APIENTRY fname fparams
#endif

#include <nv/lib/detail/assimp_functions.inc>

#undef NV_ASSIMP_FUN

}

namespace nv
{

	inline vec3 assimp_vec3_cast( const aiVector3D& v )
	{
		return vec3( v.x, v.y, v.z );
	}

	inline vec2 assimp_st_cast( const aiVector3D& v )
	{
		return vec2( v.x, 1-v.y );
	}

	inline mat4 assimp_mat4_cast( const aiMatrix4x4& m )
	{
		const float* p = reinterpret_cast<const float*>( &m );
		return math::transpose( make_mat4( p ) );
	}

	inline quat assimp_quat_cast( const aiQuaternion& q )
	{
		return quat( q.w, q.x, q.y, q.z );
	}

	inline void assimp_log_callback( const char* message, char* )
	{
		NV_LOG_DEBUG( rtrimmed( message ) );
	}

	struct assimp_log_guard
	{
		assimp_log_guard();
		~assimp_log_guard();
	};

		/* Dynamic load support */
#	if defined( NV_ASSIMP_DYNAMIC )
	bool load_assimp_library( const char* path = NV_ASSIMP_PATH );
#	else
	inline bool load_assimp_library( const char* path = "" ) { return true; }
#	endif
}


#endif // NV_LIB_ASSIMP_HH
