// Copyright (C) 2011-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/gfx/skeleton_instance.hh"

void nv::skeleton_binding::prepare( const mesh_nodes_data* node_data, const mesh_nodes_data* bone_data )
{
	if ( !m_offsets || !m_indices )
	{
		// TODO: either fixed size struct or static allocator
		hash_store< shash64, uint16 > bone_names;
		m_offsets = new mat4[bone_data->size()];
		m_indices = new sint16[node_data->size()];

		for ( nv::uint16 bi = 0; bi < bone_data->size(); ++bi )
		{
			const data_channel_set* bone = ( *bone_data )[bi];
			bone_names[bone->get_name()] = bi;
			m_offsets[bi] = bone->get_transform();
		}

		for ( uint32 n = 0; n < node_data->size(); ++n )
		{
			const data_channel_set* node = ( *node_data )[n];
			sint16 bone_id = -1;

			auto bi = bone_names.find( node->get_name() );
			if ( bi != bone_names.end() )
			{
				bone_id = sint16( bi->second );
			}
			m_indices[n] = bone_id;

		}
	}

	if ( m_key.size() == 0 )
	{
		for ( uint32 n = 0; n < node_data->size(); ++n )
			if ( ( *node_data )[n]->size() > 0 )
			{
				m_key = ( *node_data )[n]->get_interpolation_key();
				break;
			}
	}

}

void nv::skeleton_instance::animate( const mesh_nodes_data* node_data, const skeleton_binding& binding, float frame )
{
	if ( m_transform.size() > 0 )
	{
		if ( node_data->is_flat() )
		{
			animate_flat( node_data, binding, frame );
		}
		else
		{
			for ( uint32 n = 0; n < node_data->size(); ++n )
				if ( ( *node_data )[n]->get_parent_id() == -1 )
					animate_rec( node_data, binding, frame, n, mat4() );
		}
	}
}

void nv::skeleton_instance::animate_rec( const mesh_nodes_data* node_data, const skeleton_binding& binding, float frame, uint32 id, const mat4& parent ) 
{
	// TODO: fix transforms, which are now embedded,
	//       see note in assimp_loader.cc:load_node
	const data_channel_set* node = ( *node_data )[id];
	mat4 node_mat( node->get_transform() );

	if ( node->size() > 0 )
	{
		raw_channel_interpolator interpolator( node, binding.m_key );
		node_mat = interpolator.get< mat4 >( frame );
	}

	mat4 global_mat = parent * node_mat;

	sint16 bone_id = binding.m_indices[id];
	if ( bone_id >= 0 )
	{
		m_transform[bone_id] = global_mat * binding.m_offsets[bone_id];
	}

	for ( auto child : node_data->children( id ) )
	{
		animate_rec( node_data, binding, frame, child, global_mat );
	}
}

void nv::skeleton_instance::animate_flat( const mesh_nodes_data* node_data, const skeleton_binding& binding, float frame ) 
{
	for ( uint32 n = 0; n < node_data->size(); ++n )
		if ( binding.m_indices[n] >= 0 )
		{
			const data_channel_set* node = ( *node_data )[n];
			nv::mat4 node_mat( node->get_transform() );

			if ( node->size() > 0 )
			{
				raw_channel_interpolator interpolator( node, binding.m_key );
				node_mat = interpolator.get< mat4 >( frame );
			}

			sint16 bone_id = binding.m_indices[n];
			m_transform[bone_id] = node_mat * binding.m_offsets[bone_id];
		}
}
