// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file primary.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief type traits - primary traits
 */

#ifndef NV_STL_TRAITS_PRIMARY_HH
#define NV_STL_TRAITS_PRIMARY_HH

#include <nv/stl/type_traits/common.hh>

namespace nv
{

	namespace detail
	{
		template< typename T > struct is_integral_impl           : false_type {};
		template<> struct is_integral_impl< bool >               : true_type{};
		template<> struct is_integral_impl< char >               : true_type{};
		template<> struct is_integral_impl< signed char >        : true_type{};
		template<> struct is_integral_impl< unsigned char >      : true_type{};
		template<> struct is_integral_impl< wchar_t >            : true_type{};
		template<> struct is_integral_impl< signed short >       : true_type{};
		template<> struct is_integral_impl< unsigned short >     : true_type{};
		template<> struct is_integral_impl< signed int >         : true_type{};
		template<> struct is_integral_impl< unsigned int >       : true_type{};
		template<> struct is_integral_impl< signed long >        : true_type{};
		template<> struct is_integral_impl< unsigned long >      : true_type{};
		template<> struct is_integral_impl< signed long long >   : true_type{};
		template<> struct is_integral_impl< unsigned long long > : true_type{};

		template< typename T > struct is_floating_point_impl     : false_type {};
		template<> struct is_floating_point_impl< float >        : true_type{};
		template<> struct is_floating_point_impl< double >       : true_type{};
		template<> struct is_floating_point_impl< long double >  : true_type{};

		template < typename T > struct is_pointer_impl           : false_type {};
		template < typename T > struct is_pointer_impl< T* >     : true_type{};
	}

	template< typename T > struct is_void           : is_same < void,      remove_cv_t<T> > {};
	template< typename T > struct is_nullptr        : is_same < nullptr_t, remove_cv_t<T> > {};
	template< typename T > struct is_integral       : detail::is_integral_impl < remove_cv_t<T> > {};
	template< typename T > struct is_floating_point : detail::is_floating_point_impl < remove_cv_t<T> > {};

	template < typename T >           struct is_array          : false_type {};
	template < typename T >           struct is_array < T[] >  : true_type{};
	template < typename T, size_t N > struct is_array < T[N] > : true_type{};

	template < typename T >           struct is_enum  : bool_constant < __is_enum( T ) > {};
	template < typename T >           struct is_union : bool_constant < __is_union( T ) > {};
	template < typename T >           struct is_class : bool_constant < __is_class( T ) > {};

	// is_function_pointer / is_member_function pointer - extension
	// TODO: sanity check - this is very different - http://en.cppreference.com/w/cpp/types/is_function
	// TODO: "..., ..." function match version?
	// TODO: call conventions really needed?

	template < typename F > struct is_function_pointer : false_type {};
#if NV_COMPILER == NV_MSVC
	template < typename R, typename... Args > struct is_function_pointer< R( __cdecl * )( Args... ) > : true_type{};
	template < typename R, typename... Args > struct is_function_pointer< R( __stdcall * )( Args... ) > : true_type{};
	template < typename R, typename... Args > struct is_function_pointer< R( __fastcall * )( Args... ) > : true_type{};
	template < typename R, typename... Args > struct is_function_pointer< R( __vectorcall * )( Args... ) > : true_type{};
#else
	template < typename R, typename... Args > struct is_function_pointer< R( *)( Args... ) > : true_type{};
#endif

	template < typename C > struct is_member_function_pointer : false_type {};
#if NV_COMPILER == NV_MSVC
#define NV_IS_MEMFNPTR( call_conv ) \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) > : true_type{}; \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) const > : true_type{}; \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) volatile > : true_type{}; \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) const volatile > : true_type{};

NV_IS_MEMFNPTR( __thiscall )
NV_IS_MEMFNPTR( __cdecl )
NV_IS_MEMFNPTR( __stdcall )
NV_IS_MEMFNPTR( __fastcall )
NV_IS_MEMFNPTR( __vectorcall )

#undef NV_IS_MEMFNPTR
#else
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) > : true_type{};
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) const > : true_type{};
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) volatile > : true_type{};
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) const volatile > : true_type{};
#endif

	template < typename F >
	struct is_member_pointer : bool_constant < is_member_function_pointer<F>::value > {};
	template < typename C, typename F >
	struct is_member_pointer<F C::*> : true_type{};

	// TODO: check if member functions can actually match under any compiler
	template < typename T >
	struct is_pointer : bool_constant < 
		( detail::is_pointer_impl<remove_cv_t<T>>::value )
		&& !( is_member_pointer<T>::value ) > {};

	template < typename T >
	struct is_member_object_pointer : bool_constant <
		is_member_pointer<T>::value 
		&& !is_member_function_pointer<T>::value > {};

	template< typename T > struct is_function : bool_constant < is_function_pointer< remove_cv_t<T> *>::value > {};
	template< typename T > struct is_function < T& >  : false_type{};
	template< typename T > struct is_function < T&& > : false_type{};

}

#endif // NV_STL_TRAITS_PRIMARY_HH
