// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_INTERFACE_MESH_DATA_HH
#define NV_INTERFACE_MESH_DATA_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/string.hh>
#include <nv/interface/data_channel.hh>
#include <nv/gfx/animation.hh>

namespace nv
{

	struct index_u16 { uint16 index; };
	struct index_u32 { uint32 index; };

	//using mesh_data = data_channel_set;

	// TODO : attribute/property implementation and read/write on every nmd::command

	class mesh_nodes_data
	{

		static constexpr size_t MAX_CHILDREN = 7;
		class child_list : noncopyable
		{

		public:
			child_list() : m_size( 0 )
			{
				nv::raw_fill_n( m_data, MAX_CHILDREN, 0 );
			}
			void push( uint8 id )
			{
				NV_ASSERT( m_size < MAX_CHILDREN, "Range error" );
				m_data[m_size++] = id;
			}
			const uint8* begin() const { return m_data; }
			const uint8* end() const { return m_data + m_size; }
		private:
			uint8 m_size;
			uint8 m_data[MAX_CHILDREN];
		};
		static_assert( sizeof( child_list ) == 8, "Align/Padding fail for child_list" );

		friend class mesh_creator;
		friend class mesh_nodes_creator;
	public:
		typedef vector< data_channel_set* > storage;
		typedef storage::const_iterator const_iterator;

		explicit mesh_nodes_data( shash64 name )
			: m_name( name ), m_frame_rate(0), m_frame_count(0), m_flat( false )
		{
		}

		explicit mesh_nodes_data( shash64 name, uint16 a_fps, uint16 a_frames, bool a_flat )
			: m_name( name ), m_frame_rate(a_fps), m_frame_count(a_frames), m_flat( a_flat )
		{
		}

		void append( data_channel_set* set )
		{
			m_data.push_back( set );
		}

		void initialize()
		{
			m_children.resize( m_data.size() );
			for ( uint8 n = 0; n < uint8( m_data.size() ); ++n )
			{
				const data_channel_set* node = m_data[n];
				if ( node->get_parent_id() != -1 )
				{
					m_children[node->get_parent_id()].push( n );
				}
			}
		}

		const child_list& children( size_t i ) const
		{
			return m_children[i];
		}

		size_t size() const { return m_data.size(); }
		bool empty() const { return m_data.empty(); }

		bool is_animated( size_t i ) const
		{
			if ( i >= m_data.size() ) return false;
			return ( m_data[i]->size() > 0 );
		}

		bool is_animated() const
		{
			for ( auto data : m_data )
			{
				if ( data->size() > 0 ) return true;
			}
			return false;
		}

		const data_channel_set* get_by_hash( shash64 h ) const
		{
			for ( auto data : m_data )
			{
				if ( data->get_name() == h ) return data;
			}
			return nullptr;
		}

		int get_index_by_hash( shash64 h ) const
		{
			for ( uint32 i = 0; i < m_data.size(); ++i )
			{
				if ( m_data[ i ]->get_name() == h )
					return int( i );
			}
			return -1;
		}

		const data_channel_set* operator[]( size_t i ) const
		{
			return m_data[i];
		}

		const_iterator begin() const { return m_data.begin(); }
		const_iterator end() const { return m_data.end(); }

		bool is_flat() const { return m_flat; }
		uint16 get_fps() const { return m_frame_rate; }
		uint16 get_frame_count() const { return m_frame_count; }
		shash64 get_name() const { return m_name; }

		~mesh_nodes_data()
		{
			for ( auto data : m_data ) delete data;
		}

	private:
		vector< data_channel_set* > m_data;
		vector< child_list >        m_children;
		shash64 m_name;
		uint16  m_frame_rate;
		uint16  m_frame_count;
		bool    m_flat;
	};

	class mesh_data_pack
	{
		friend class mesh_creator;
	public:
		explicit mesh_data_pack( uint32 a_count, data_channel_set* a_meshes, mesh_nodes_data* a_nodes = nullptr )
		{
			m_count  = a_count;
			m_meshes = a_meshes;
			m_nodes  = a_nodes;
		}
		const data_channel_set* get_mesh( uint32 index ) const
		{
			if ( index >= m_count ) return nullptr;
			return &m_meshes[ index ];
		}
		const data_channel_set* get_mesh_by_hash( shash64 h ) const
		{
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_meshes[i].get_name() == h )
					return &m_meshes[i];
			}
			return nullptr;
		}
		uint32 get_count() const { return m_count; }
		data_channel_set* release_meshes() { data_channel_set* meshes = m_meshes; m_meshes = nullptr; return meshes; }
		mesh_nodes_data* release_nodes() { mesh_nodes_data* nodes = m_nodes; m_nodes = nullptr; return nodes; }
		const mesh_nodes_data* get_nodes() const { return m_nodes; }
		uint32 get_node_count() const { return m_nodes ? m_nodes->size() : 0; }
		~mesh_data_pack()
		{
			delete[] m_meshes;
			delete   m_nodes;
		}
	private:
		uint32            m_count;
		data_channel_set* m_meshes;
		mesh_nodes_data*  m_nodes;
	};
}

#endif // NV_INTERFACE_MESH_DATA_HH
