// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file resource_manager.hh
 * @author Kornel Kisielewicz
 * @brief engine resource manager
 */

#ifndef NV_ENGINE_RESOURCE_MANAGER_HH
#define NV_ENGINE_RESOURCE_MANAGER_HH

#include <nv/common.hh>
#include <nv/interface/context.hh>
#include <nv/core/resource.hh>
#include <nv/lua/lua_state.hh>
#include <nv/stl/hash_store.hh>
#include <nv/stl/vector.hh>

namespace nv
{

	template < typename T, bool Heap = true >
	struct resource_storage_policy;

	template < typename T >
	struct resource_storage_policy< T, true >
	{
		typedef T* type;
		static void free( T* value ) { delete value; }
		static T* to_pointer( T* value ) { return value; }
	};

	template < typename T >
	struct resource_storage_policy< T, false >
	{
		typedef T type;
		static void free( T ) {}
		static T* to_pointer( T& value ) { return &value; }
	};

	class resource_system;

	class custom_resource_manager_base : public resource_handler
	{
	public:
		custom_resource_manager_base() {}
		virtual void clear() = 0;
		virtual bool load_resource( const string_view& id ) = 0;
	protected:
		virtual void unlock( resource_id, resource_type_id ) {};
		virtual void release( resource_id, resource_type_id ) {};
	};

	class lua_resource_manager_base : public resource_handler
	{
	public:
		lua_resource_manager_base() : m_lua( nullptr ) {}
		void initialize( lua::state* state );
		virtual string_view get_storage_name() const = 0;
		virtual string_view get_resource_name() const = 0;
		virtual void clear() = 0;
		void load_all();
		bool load_resource( const string_view& id );
		virtual ~lua_resource_manager_base() {}
	protected:
		virtual bool load_resource( lua::table_guard& table, shash64 id ) = 0;
		//		virtual const void* lock( resource_id id, resource_type_id );
		virtual void unlock( resource_id, resource_type_id ) {};
		virtual void release( resource_id, resource_type_id ) {};

		lua::state* m_lua;
	};

	template < typename T, bool Heap = true, typename Base = custom_resource_manager_base >
	class custom_resource_manager : public Base
	{
	public:
		typedef resource_storage_policy< T, Heap > policy_type;
		typedef T                          value_type;
		typedef resource< T >              resource_type;
		typedef typename policy_type::type stored_type;

		custom_resource_manager() {}
		resource_type get( const string_view& id )
		{
			auto m = m_store.find( id );
			if ( m != m_store.end() )
			{
				return create< T >( id );
			}
			else
			{
				if ( this->load_resource( id ) )
				{
					return create< T >( id );
				}
			}
			// NV_ASSERT( false, "resource_manager.get failed!" );
			return resource_type();
		}

		resource_type get( uint64 id )
		{
			auto m = m_store.find( shash64( id ) );
			if ( m != m_store.end() )
			{
				return create< T >( shash64( id ) );
			}
			// NV_ASSERT( false, "resource_manager.get failed!" );
			return resource_type();
		}


		virtual void clear()
		{
			for ( auto data : m_store )
			{
				release( data.second );
				policy_type::free( data.second );
			}
			m_store.clear();
		}

		virtual ~custom_resource_manager()
		{
			clear();
		}
	protected:
		virtual const void* lock( resource_id id, resource_type_id )
		{
			auto m = m_store.find( id );
			return m != m_store.end() ? policy_type::to_pointer( m->second ) : nullptr;
		}

		virtual void release( stored_type ) {}

		void add( stored_type resource, shash64 id )
		{
			m_store[id] = resource;
		}

		hash_store< shash64, stored_type > m_store;
	};

	template < typename T, bool Heap = true >
	using lua_resource_manager = custom_resource_manager< T, Heap, lua_resource_manager_base >;

}

#endif // NV_ENGINE_RESOURCE_MANAGER_HH
